import { useState, useEffect } from "react";
import { Card } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { PatchEngine } from "./patchEngine";
import { supabase } from "@/integrations/supabase/client";
import { History, Undo2, FileCode, Calendar, CheckCircle2, XCircle, Clock } from "lucide-react";
import { format } from "date-fns";

type PatchStatus = 'pending' | 'applied' | 'rolled_back';

interface Patch {
  id: string;
  file_path: string;
  status: PatchStatus;
  created_at: string;
  applied_at: string | null;
  patched_code: string;
  original_code: string | null;
  fix_plan_id: string | null;
}

export default function PatchHistory() {
  const [patches, setPatches] = useState<Patch[]>([]);
  const [loading, setLoading] = useState(true);
  const [rollingBack, setRollingBack] = useState<string | null>(null);

  const fetchPatches = async () => {
    setLoading(true);
    const data = await PatchEngine.getPatchHistory();
    setPatches(data as Patch[]);
    setLoading(false);
  };

  useEffect(() => {
    fetchPatches();

    // Subscribe to realtime changes
    const channel = supabase
      .channel('patch_history_changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'code_patches'
        },
        () => {
          fetchPatches();
        }
      )
      .subscribe();

    return () => {
      supabase.removeChannel(channel);
    };
  }, []);

  const handleRollback = async (patchId: string) => {
    setRollingBack(patchId);
    const success = await PatchEngine.rollbackPatch(patchId);
    if (success) {
      await fetchPatches();
    }
    setRollingBack(null);
  };

  const getStatusBadge = (status: PatchStatus) => {
    switch (status) {
      case 'applied':
        return (
          <Badge variant="default" className="gap-1">
            <CheckCircle2 className="w-3 h-3" />
            Applied
          </Badge>
        );
      case 'rolled_back':
        return (
          <Badge variant="secondary" className="gap-1">
            <Undo2 className="w-3 h-3" />
            Rolled Back
          </Badge>
        );
      case 'pending':
        return (
          <Badge variant="outline" className="gap-1">
            <Clock className="w-3 h-3" />
            Pending
          </Badge>
        );
      default:
        return null;
    }
  };

  if (loading) {
    return (
      <Card className="p-6">
        <div className="flex items-center gap-2 mb-4">
          <History className="w-5 h-5 text-primary" />
          <h3 className="text-lg font-semibold">প্যাচ ইতিহাস • Patch History</h3>
        </div>
        <p className="text-muted-foreground">লোড হচ্ছে • Loading...</p>
      </Card>
    );
  }

  return (
    <Card className="p-6">
      <div className="flex items-center justify-between mb-4">
        <div className="flex items-center gap-2">
          <History className="w-5 h-5 text-primary" />
          <h3 className="text-lg font-semibold">প্যাচ ইতিহাস • Patch History</h3>
          <Badge variant="secondary">{patches.length} total</Badge>
        </div>
      </div>

      {patches.length === 0 ? (
        <p className="text-muted-foreground text-center py-8">
          কোনো প্যাচ পাওয়া যায়নি • No patches found
        </p>
      ) : (
        <ScrollArea className="h-[500px]">
          <div className="space-y-3">
            {patches.map((patch) => (
              <Card key={patch.id} className="p-4 bg-muted/30">
                <div className="flex items-start justify-between gap-4">
                  <div className="flex-1 space-y-2">
                    <div className="flex items-center gap-2">
                      <FileCode className="w-4 h-4 text-primary" />
                      <code className="text-sm font-mono">{patch.file_path}</code>
                      {getStatusBadge(patch.status)}
                    </div>
                    
                    <div className="flex items-center gap-4 text-xs text-muted-foreground">
                      <div className="flex items-center gap-1">
                        <Calendar className="w-3 h-3" />
                        Created: {format(new Date(patch.created_at), 'PPp')}
                      </div>
                      {patch.applied_at && (
                        <div className="flex items-center gap-1">
                          <CheckCircle2 className="w-3 h-3" />
                          Applied: {format(new Date(patch.applied_at), 'PPp')}
                        </div>
                      )}
                    </div>

                    {patch.original_code && (
                      <details className="text-xs">
                        <summary className="cursor-pointer text-primary hover:underline">
                          কোড দেখুন • View Code
                        </summary>
                        <div className="mt-2 space-y-2">
                          <div>
                            <p className="font-semibold mb-1">Original:</p>
                            <pre className="bg-background p-2 rounded text-[10px] overflow-x-auto">
                              {patch.original_code.substring(0, 200)}
                              {patch.original_code.length > 200 && '...'}
                            </pre>
                          </div>
                          <div>
                            <p className="font-semibold mb-1">Patched:</p>
                            <pre className="bg-background p-2 rounded text-[10px] overflow-x-auto">
                              {patch.patched_code.substring(0, 200)}
                              {patch.patched_code.length > 200 && '...'}
                            </pre>
                          </div>
                        </div>
                      </details>
                    )}
                  </div>

                  <div>
                    {patch.status === 'applied' && patch.original_code && (
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleRollback(patch.id)}
                        disabled={rollingBack === patch.id}
                        className="gap-2"
                      >
                        <Undo2 className="w-4 h-4" />
                        {rollingBack === patch.id ? 'রোলব্যাক করা হচ্ছে...' : 'Rollback'}
                      </Button>
                    )}
                    {patch.status === 'rolled_back' && (
                      <Badge variant="secondary" className="gap-1">
                        <XCircle className="w-3 h-3" />
                        Reverted
                      </Badge>
                    )}
                  </div>
                </div>
              </Card>
            ))}
          </div>
        </ScrollArea>
      )}
    </Card>
  );
}
