import { useState, useEffect } from "react";
import { Card } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { PatchEngine } from "./patchEngine";
import { supabase } from "@/integrations/supabase/client";
import { FileCode, CheckCircle2, XCircle, Eye } from "lucide-react";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from "@/components/ui/dialog";

interface Patch {
  id: string;
  file_path: string;
  status: string;
  created_at: string;
  patched_code: string;
  original_code: string | null;
  fix_plan_id: string | null;
}

export default function PendingPatches() {
  const [patches, setPatches] = useState<Patch[]>([]);
  const [loading, setLoading] = useState(true);
  const [processing, setProcessing] = useState<string | null>(null);
  const [previewPatch, setPreviewPatch] = useState<Patch | null>(null);

  const fetchPatches = async () => {
    setLoading(true);
    const data = await PatchEngine.getPendingPatches();
    setPatches(data as Patch[]);
    setLoading(false);
  };

  useEffect(() => {
    fetchPatches();

    // Subscribe to realtime changes
    const channel = supabase
      .channel('pending_patches_changes')
      .on(
        'postgres_changes',
        {
          event: '*',
          schema: 'public',
          table: 'code_patches'
        },
        () => {
          fetchPatches();
        }
      )
      .subscribe();

    return () => {
      supabase.removeChannel(channel);
    };
  }, []);

  const handleApply = async (patchId: string) => {
    setProcessing(patchId);
    const success = await PatchEngine.applyPatch(patchId);
    if (success) {
      await fetchPatches();
    }
    setProcessing(null);
  };

  const handleReject = async (patchId: string) => {
    setProcessing(patchId);
    const success = await PatchEngine.rejectPatch(patchId);
    if (success) {
      await fetchPatches();
    }
    setProcessing(null);
  };

  if (loading) {
    return (
      <Card className="p-6">
        <div className="flex items-center gap-2 mb-4">
          <FileCode className="w-5 h-5 text-primary" />
          <h3 className="text-lg font-semibold">প্যাচ অপেক্ষমাণ • Pending Patches</h3>
        </div>
        <p className="text-muted-foreground">লোড হচ্ছে • Loading...</p>
      </Card>
    );
  }

  return (
    <>
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center gap-2">
            <FileCode className="w-5 h-5 text-primary" />
            <h3 className="text-lg font-semibold">প্যাচ অপেক্ষমাণ • Pending Patches</h3>
            <Badge variant="secondary">{patches.length} pending</Badge>
          </div>
        </div>

        {patches.length === 0 ? (
          <p className="text-muted-foreground text-center py-8">
            কোনো অপেক্ষমাণ প্যাচ নেই • No pending patches
          </p>
        ) : (
          <ScrollArea className="h-[400px]">
            <div className="space-y-3">
              {patches.map((patch) => (
                <Card key={patch.id} className="p-4 bg-muted/30">
                  <div className="flex items-start justify-between gap-4">
                    <div className="flex-1 space-y-2">
                      <div className="flex items-center gap-2">
                        <FileCode className="w-4 h-4 text-primary" />
                        <code className="text-sm font-mono">{patch.file_path}</code>
                        <Badge variant="outline">Pending</Badge>
                      </div>
                      
                      <div className="text-xs text-muted-foreground">
                        Created: {new Date(patch.created_at).toLocaleString('en-US', { 
                          dateStyle: 'medium', 
                          timeStyle: 'short' 
                        })}
                      </div>

                      <Button
                        size="sm"
                        variant="ghost"
                        onClick={() => setPreviewPatch(patch)}
                        className="gap-2"
                      >
                        <Eye className="w-4 h-4" />
                        কোড দেখুন • Preview Code
                      </Button>
                    </div>

                    <div className="flex gap-2">
                      <Button
                        size="sm"
                        onClick={() => handleApply(patch.id)}
                        disabled={processing === patch.id}
                        className="gap-2"
                      >
                        <CheckCircle2 className="w-4 h-4" />
                        {processing === patch.id ? 'প্রয়োগ হচ্ছে...' : 'Apply'}
                      </Button>
                      <Button
                        size="sm"
                        variant="outline"
                        onClick={() => handleReject(patch.id)}
                        disabled={processing === patch.id}
                        className="gap-2"
                      >
                        <XCircle className="w-4 h-4" />
                        Reject
                      </Button>
                    </div>
                  </div>
                </Card>
              ))}
            </div>
          </ScrollArea>
        )}
      </Card>

      {/* Preview Dialog */}
      <Dialog open={!!previewPatch} onOpenChange={() => setPreviewPatch(null)}>
        <DialogContent className="max-w-4xl max-h-[80vh]">
          <DialogHeader>
            <DialogTitle>কোড প্যাচ প্রিভিউ • Code Patch Preview</DialogTitle>
            <DialogDescription>
              <code className="text-sm">{previewPatch?.file_path}</code>
            </DialogDescription>
          </DialogHeader>
          
          <ScrollArea className="h-[500px]">
            <div className="space-y-4">
              {previewPatch?.original_code && (
                <div>
                  <p className="font-semibold mb-2 text-sm">মূল কোড • Original Code:</p>
                  <pre className="bg-muted p-4 rounded text-xs overflow-x-auto">
                    {previewPatch.original_code}
                  </pre>
                </div>
              )}
              
              <div>
                <p className="font-semibold mb-2 text-sm">নতুন কোড • Patched Code:</p>
                <pre className="bg-muted p-4 rounded text-xs overflow-x-auto">
                  {previewPatch?.patched_code}
                </pre>
              </div>
            </div>
          </ScrollArea>

          <DialogFooter>
            <Button variant="outline" onClick={() => setPreviewPatch(null)}>
              বন্ধ করুন • Close
            </Button>
            {previewPatch && (
              <>
                <Button
                  variant="outline"
                  onClick={() => {
                    handleReject(previewPatch.id);
                    setPreviewPatch(null);
                  }}
                  className="gap-2"
                >
                  <XCircle className="w-4 h-4" />
                  Reject
                </Button>
                <Button
                  onClick={() => {
                    handleApply(previewPatch.id);
                    setPreviewPatch(null);
                  }}
                  className="gap-2"
                >
                  <CheckCircle2 className="w-4 h-4" />
                  Apply Patch
                </Button>
              </>
            )}
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </>
  );
}
