import { supabase } from "@/integrations/supabase/client";
import { toast } from "sonner";

export interface GenerateTestContext {
  errorId: string;
  errorMessage: string;
  filePath: string | null;
  issueType: string;
  severity: string;
  details: any;
}

export class TestGenerator {
  /**
   * Generate unit tests from an error using AI analysis
   */
  static async generateFromError(context: GenerateTestContext): Promise<string | null> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error('Not authenticated');

      toast.info('টেস্ট তৈরি করা হচ্ছে • Generating tests...');

      // Call the AI analyze function to generate tests
      const { data, error } = await supabase.functions.invoke('core-brain-analyze', {
        body: {
          type: 'test_generation',
          context: {
            error_message: context.errorMessage,
            file_path: context.filePath,
            issue_type: context.issueType,
            severity: context.severity,
            details: context.details
          }
        }
      });

      if (error) throw error;

      // Determine test file path
      const testFilePath = context.filePath 
        ? context.filePath.replace(/\.(tsx?|jsx?)$/, '.test.$1')
        : `src/tests/${context.issueType.replace(/\s+/g, '-').toLowerCase()}.test.ts`;

      // Save the generated test
      const { data: test, error: insertError } = await supabase
        .from('generated_tests')
        .insert({
          scan_result_id: context.errorId,
          test_file_path: testFilePath,
          test_code: data.test_code || '',
          test_framework: data.framework || 'vitest',
          description: data.description || `Test for: ${context.errorMessage}`,
          status: 'pending',
          created_by: user.id
        })
        .select()
        .single();

      if (insertError) throw insertError;

      await supabase.from('core_brain_logs').insert({
        type: 'success',
        message: `Unit test generated for error: ${context.errorMessage.substring(0, 50)}...`,
        details: { test_id: test.id, scan_result_id: context.errorId }
      });

      toast.success('টেস্ট তৈরি সম্পন্ন! • Test generated successfully!');
      return test.id;
    } catch (error) {
      console.error('Error generating test:', error);
      toast.error('টেস্ট তৈরি ব্যর্থ • Failed to generate test');
      return null;
    }
  }

  /**
   * Apply a generated test (marks as approved and applied)
   */
  static async applyTest(testId: string): Promise<boolean> {
    try {
      const { data: test } = await supabase
        .from('generated_tests')
        .select('*')
        .eq('id', testId)
        .single();

      if (!test) {
        throw new Error('Test not found');
      }

      if (test.status === 'approved') {
        toast.info('এই টেস্ট ইতিমধ্যে অনুমোদিত • Test already approved');
        return false;
      }

      // Mark test as approved
      const { error: updateError } = await supabase
        .from('generated_tests')
        .update({
          status: 'approved',
          applied_at: new Date().toISOString()
        })
        .eq('id', testId);

      if (updateError) throw updateError;

      // Log the application
      await supabase.from('core_brain_logs').insert({
        type: 'success',
        message: `Unit test approved: ${test.test_file_path}`,
        details: { test_id: testId, file_path: test.test_file_path }
      });

      toast.success('টেস্ট অনুমোদিত! • Test approved successfully!');
      return true;
    } catch (error) {
      console.error('Error applying test:', error);
      toast.error('টেস্ট অনুমোদন ব্যর্থ • Failed to approve test');
      return false;
    }
  }

  /**
   * Reject/delete a generated test
   */
  static async rejectTest(testId: string): Promise<boolean> {
    try {
      const { error } = await supabase
        .from('generated_tests')
        .update({ status: 'rejected' })
        .eq('id', testId);

      if (error) throw error;

      await supabase.from('core_brain_logs').insert({
        type: 'warning',
        message: `Unit test rejected`,
        details: { test_id: testId }
      });

      toast.info('টেস্ট বাতিল • Test rejected');
      return true;
    } catch (error) {
      console.error('Error rejecting test:', error);
      toast.error('টেস্ট বাতিল ব্যর্থ • Failed to reject test');
      return false;
    }
  }

  /**
   * Get all pending tests
   */
  static async getPendingTests() {
    const { data, error } = await supabase
      .from('generated_tests')
      .select('*, scan_results(*)')
      .eq('status', 'pending')
      .order('created_at', { ascending: false });

    if (error) {
      console.error('Error fetching pending tests:', error);
      return [];
    }

    return data || [];
  }

  /**
   * Get test history (all tests)
   */
  static async getTestHistory() {
    const { data, error } = await supabase
      .from('generated_tests')
      .select('*, scan_results(*)')
      .order('created_at', { ascending: false });

    if (error) {
      console.error('Error fetching test history:', error);
      return [];
    }

    return data || [];
  }
}
