import { BrainRouter } from "./BrainRouter";
import { BrainMemory } from "./BrainMemory";
import { BrainIdentity } from "./BrainIdentity";
import { BrainTools } from "./BrainTools";

export interface BrainInput {
  userId: string;
  role: "owner" | "child" | "guest" | "user";
  channel: string;
  text: string;
  locale: string;
  context?: any;
}

export interface BrainOutput {
  replyText: string;
  suggestedModule?: string;
  actions?: Array<{ type: string; data: any }>;
  memoryWrites?: Array<{ key: string; value: any }>;
  emotion?: string;
  confidence?: number;
}

export class BrainEngine {
  private router: BrainRouter;
  private memory: BrainMemory;
  private identity: BrainIdentity;
  private tools: BrainTools;

  constructor() {
    this.router = new BrainRouter();
    this.memory = new BrainMemory();
    this.identity = new BrainIdentity();
    this.tools = new BrainTools();
  }

  async process(input: BrainInput): Promise<BrainOutput> {
    try {
      // 1. Load user memory
      const userMemory = await this.memory.getUserMemory(input.userId);

      // 2. Check identity rules first
      const identityMatch = await this.identity.checkIdentityRules(input.text, input.locale);
      if (identityMatch) {
        return {
          replyText: identityMatch.answer,
          emotion: "confident",
          confidence: 1.0,
        };
      }

      // 3. Apply safety filters based on role
      const safetyCheck = await this.applySafetyFilter(input);
      if (!safetyCheck.safe) {
        return {
          replyText: safetyCheck.message,
          emotion: "caring",
          confidence: 1.0,
        };
      }

      // 4. Classify intent and route to module
      const routing = await this.router.classifyAndRoute(input.text, input.locale);

      // 5. Build context for AI
      const context = this.buildContext(input, userMemory, routing);

      // 6. Get AI response (using tools if needed)
      const aiResponse = await this.getAIResponse(context, input, routing);

      // 7. Extract actions and memory updates
      const actions = this.extractActions(aiResponse, routing);
      const memoryWrites = this.extractMemoryUpdates(aiResponse, input);

      // 8. Save to memory
      if (memoryWrites.length > 0) {
        await this.memory.saveMemory(input.userId, memoryWrites);
      }

      return {
        replyText: aiResponse.text,
        suggestedModule: routing.suggestedModule,
        actions,
        memoryWrites,
        emotion: aiResponse.emotion || "friendly",
        confidence: routing.confidence,
      };
    } catch (error) {
      console.error("Brain processing error:", error);
      return {
        replyText: this.getErrorMessage(input.locale),
        emotion: "apologetic",
        confidence: 0.5,
      };
    }
  }

  private async applySafetyFilter(input: BrainInput): Promise<{ safe: boolean; message: string }> {
    // Child/Ava Mode - strict safety
    if (input.role === "child") {
      const harmful = await this.detectHarmfulContent(input.text);
      if (harmful) {
        return {
          safe: false,
          message: input.locale === "bn"
            ? "দুঃখিত, আমি এই বিষয়ে সাহায্য করতে পারি না। চলো অন্য কিছু নিয়ে কথা বলি!"
            : "I'm sorry, I can't help with that. Let's talk about something else!",
        };
      }
    }

    // Owner Mode - less restrictions but still ethical
    if (input.role === "owner") {
      // Minimal filtering for owner
      return { safe: true, message: "" };
    }

    // Guest/User Mode - standard safety
    const harmful = await this.detectHarmfulContent(input.text);
    if (harmful) {
      return {
        safe: false,
        message: input.locale === "bn"
          ? "দুঃখিত, আমি এই ধরনের বিষয়ে সাহায্য করতে পারি না।"
          : "I'm sorry, I cannot help with that topic.",
      };
    }

    return { safe: true, message: "" };
  }

  private async detectHarmfulContent(text: string): Promise<boolean> {
    // Simple keyword-based detection (can be enhanced with AI)
    const harmfulPatterns = [
      /\b(violence|weapon|drug|illegal)\b/i,
      /\b(hack|crack|pirate)\b/i,
    ];

    return harmfulPatterns.some(pattern => pattern.test(text));
  }

  private buildContext(input: BrainInput, memory: any, routing: any): string {
    let context = `User Role: ${input.role}\n`;
    context += `Language: ${input.locale}\n`;
    
    if (routing.suggestedModule) {
      context += `Suggested Module: ${routing.suggestedModule}\n`;
    }

    if (memory && memory.preferences) {
      context += `User Preferences: ${JSON.stringify(memory.preferences)}\n`;
    }

    if (memory && memory.recentTopics) {
      context += `Recent Topics: ${memory.recentTopics.join(", ")}\n`;
    }

    return context;
  }

  private async getAIResponse(
    context: string,
    input: BrainInput,
    routing: any
  ): Promise<{ text: string; emotion?: string }> {
    // This will be implemented by calling the edge function
    // For now, return a placeholder
    return {
      text: "I'm processing your request...",
      emotion: "thinking",
    };
  }

  private extractActions(response: any, routing: any): Array<{ type: string; data: any }> {
    const actions = [];

    if (routing.suggestedModule) {
      actions.push({
        type: "suggest_module",
        data: { module: routing.suggestedModule },
      });
    }

    return actions;
  }

  private extractMemoryUpdates(response: any, input: BrainInput): Array<{ key: string; value: any }> {
    const updates = [];

    // Always save recent message
    updates.push({
      key: "last_message",
      value: { text: input.text, timestamp: Date.now() },
    });

    return updates;
  }

  private getErrorMessage(locale: string): string {
    return locale === "bn"
      ? "দুঃখিত, আমার কিছু সমস্যা হয়েছে। আবার চেষ্টা করুন।"
      : "Sorry, I encountered an error. Please try again.";
  }
}
