import { supabase } from "@/integrations/supabase/client";

export interface IdentityRule {
  id: string;
  question_pattern: string;
  fixed_answer: string;
  language: string;
  priority: number;
}

export class BrainIdentity {
  private rules: IdentityRule[] = [];
  private rulesLoaded: boolean = false;
  private lastLoadTime: number = 0;
  private cacheExpiry: number = 10 * 60 * 1000; // 10 minutes

  // Owner information - NEVER SHARE FULL ADDRESS
  private readonly ownerInfo = {
    name_en: "Engineer S M Nazrul Islam Tasnim",
    name_bn: "ইঞ্জিনিয়ার এস এম নজরুল ইসলাম তাসনিম",
    role: "Creator and Owner of Smart Friend",
    company: "Smart Seba HQ",
    location: "Bangladesh", // Only country, never full address
  };

  async checkIdentityRules(text: string, locale: string): Promise<{ answer: string } | null> {
    try {
      // Load rules if needed
      if (!this.rulesLoaded || Date.now() - this.lastLoadTime > this.cacheExpiry) {
        await this.loadRules();
      }

      // Check for identity questions
      const lowerText = text.toLowerCase();

      // Filter rules by language (use matching locale or 'all')
      const relevantRules = this.rules.filter(
        rule => rule.language === locale || rule.language === "all"
      );

      // Sort by priority (higher first)
      relevantRules.sort((a, b) => b.priority - a.priority);

      // Check each rule
      for (const rule of relevantRules) {
        if (this.matchesPattern(lowerText, rule.question_pattern)) {
          return { answer: rule.fixed_answer };
        }
      }

      // Fallback identity checks (in case database is empty)
      return this.checkBuiltInIdentityRules(lowerText, locale);
    } catch (error) {
      console.error("Error checking identity rules:", error);
      return null;
    }
  }

  private async loadRules(): Promise<void> {
    try {
      const { data, error } = await supabase
        .from("identity_rules")
        .select("*")
        .order("priority", { ascending: false });

      if (error) throw error;

      this.rules = data || [];
      this.rulesLoaded = true;
      this.lastLoadTime = Date.now();
    } catch (error) {
      console.error("Error loading identity rules:", error);
      // Use built-in rules as fallback
      this.rules = [];
    }
  }

  private matchesPattern(text: string, pattern: string): boolean {
    try {
      // Simple keyword matching (can be enhanced with regex)
      const keywords = pattern.toLowerCase().split("|");
      return keywords.some(keyword => text.includes(keyword.trim()));
    } catch (error) {
      return false;
    }
  }

  private checkBuiltInIdentityRules(text: string, locale: string): { answer: string } | null {
    // Who created you?
    if (this.matchesPattern(text, "who created|who made|তৈরি করেছে|কে বানিয়েছে|creator|maker")) {
      return {
        answer: locale === "bn"
          ? `আমাকে তৈরি করেছেন ${this.ownerInfo.name_bn}। তিনি ${this.ownerInfo.company} এর প্রতিষ্ঠাতা এবং একজন দক্ষ ইঞ্জিনিয়ার।`
          : `I was created by ${this.ownerInfo.name_en}, the founder of ${this.ownerInfo.company} and a skilled engineer.`,
      };
    }

    // Who is your owner?
    if (this.matchesPattern(text, "your owner|তোমার মালিক|owner|মালিক")) {
      return {
        answer: locale === "bn"
          ? `আমার মালিক হলেন ${this.ownerInfo.name_bn}। তিনি স্মার্ট ফ্রেন্ড প্রকল্পের প্রধান।`
          : `My owner is ${this.ownerInfo.name_en}, the lead of Smart Friend project.`,
      };
    }

    // What's your name?
    if (this.matchesPattern(text, "your name|তোমার নাম|what are you called|তোমাকে কি বলে")) {
      return {
        answer: locale === "bn"
          ? "আমার নাম স্মার্ট ফ্রেন্ড। আমি একটি বুদ্ধিমান সহায়ক যা আপনাকে বিভিন্ন কাজে সাহায্য করতে পারে।"
          : "My name is Smart Friend. I'm an intelligent assistant that can help you with various tasks.",
      };
    }

    // Where are you from?
    if (this.matchesPattern(text, "where are you from|তুমি কোথায় থেকে|where is|location")) {
      return {
        answer: locale === "bn"
          ? `আমি বাংলাদেশে তৈরি হয়েছি ${this.ownerInfo.company} টিম দ্বারা।`
          : `I was developed in ${this.ownerInfo.location} by the ${this.ownerInfo.company} team.`,
      };
    }

    // Privacy - never share full address
    if (this.matchesPattern(text, "address|ঠিকানা|where does|contact")) {
      return {
        answer: locale === "bn"
          ? "আমি ব্যক্তিগত ঠিকানা শেয়ার করতে পারি না। আপনি Smart Seba HQ এর মাধ্যমে যোগাযোগ করতে পারেন।"
          : "I cannot share personal addresses. You can reach out through Smart Seba HQ official channels.",
      };
    }

    return null;
  }

  getOwnerInfo(): typeof this.ownerInfo {
    return { ...this.ownerInfo };
  }

  async addIdentityRule(
    questionPattern: string,
    fixedAnswer: string,
    language: string,
    priority: number = 100
  ): Promise<void> {
    try {
      const { error } = await supabase
        .from("identity_rules")
        .insert({
          question_pattern: questionPattern,
          fixed_answer: fixedAnswer,
          language,
          priority,
        });

      if (error) throw error;

      // Reload rules
      await this.loadRules();
    } catch (error) {
      console.error("Error adding identity rule:", error);
      throw error;
    }
  }
}
