export interface UserMemory {
  userId: string;
  shortTerm: Array<{ timestamp: number; content: string; type: string }>;
  longTerm: {
    preferences?: any;
    recentTopics?: string[];
    learningStyle?: string;
    favoriteModules?: string[];
  };
}

export class BrainMemory {
  private memoryCache: Map<string, UserMemory> = new Map();
  private readonly maxShortTermItems = 50;
  private readonly shortTermExpiryMs = 24 * 60 * 60 * 1000; // 24 hours

  async getUserMemory(userId: string): Promise<UserMemory> {
    // Check cache first
    if (this.memoryCache.has(userId)) {
      return this.memoryCache.get(userId)!;
    }

    // Load from localStorage (simulated long-term storage)
    const stored = this.loadFromStorage(userId);
    if (stored) {
      this.memoryCache.set(userId, stored);
      return stored;
    }

    // Create new memory
    const newMemory: UserMemory = {
      userId,
      shortTerm: [],
      longTerm: {
        preferences: {},
        recentTopics: [],
        favoriteModules: [],
      },
    };

    this.memoryCache.set(userId, newMemory);
    return newMemory;
  }

  async saveMemory(userId: string, updates: Array<{ key: string; value: any }>): Promise<void> {
    const memory = await this.getUserMemory(userId);

    // Process updates
    updates.forEach(update => {
      if (update.key === "last_message") {
        // Add to short-term memory
        memory.shortTerm.push({
          timestamp: Date.now(),
          content: update.value.text,
          type: "message",
        });

        // Trim short-term memory if too long
        if (memory.shortTerm.length > this.maxShortTermItems) {
          memory.shortTerm = memory.shortTerm.slice(-this.maxShortTermItems);
        }
      } else if (update.key.startsWith("preference_")) {
        // Store in long-term preferences
        const prefKey = update.key.replace("preference_", "");
        memory.longTerm.preferences = memory.longTerm.preferences || {};
        memory.longTerm.preferences[prefKey] = update.value;
      } else if (update.key === "topic") {
        // Add to recent topics
        memory.longTerm.recentTopics = memory.longTerm.recentTopics || [];
        if (!memory.longTerm.recentTopics.includes(update.value)) {
          memory.longTerm.recentTopics.unshift(update.value);
          // Keep only last 10 topics
          memory.longTerm.recentTopics = memory.longTerm.recentTopics.slice(0, 10);
        }
      } else if (update.key === "favorite_module") {
        // Add to favorite modules
        memory.longTerm.favoriteModules = memory.longTerm.favoriteModules || [];
        if (!memory.longTerm.favoriteModules.includes(update.value)) {
          memory.longTerm.favoriteModules.push(update.value);
        }
      }
    });

    // Clean expired short-term memories
    const now = Date.now();
    memory.shortTerm = memory.shortTerm.filter(
      item => now - item.timestamp < this.shortTermExpiryMs
    );

    // Update cache
    this.memoryCache.set(userId, memory);

    // Persist to storage
    this.saveToStorage(userId, memory);
  }

  getRecentContext(userId: string, limit: number = 5): string[] {
    const memory = this.memoryCache.get(userId);
    if (!memory) return [];

    return memory.shortTerm
      .slice(-limit)
      .map(item => item.content);
  }

  getUserPreferences(userId: string): any {
    const memory = this.memoryCache.get(userId);
    return memory?.longTerm.preferences || {};
  }

  clearShortTermMemory(userId: string): void {
    const memory = this.memoryCache.get(userId);
    if (memory) {
      memory.shortTerm = [];
      this.memoryCache.set(userId, memory);
      this.saveToStorage(userId, memory);
    }
  }

  private loadFromStorage(userId: string): UserMemory | null {
    try {
      const stored = localStorage.getItem(`brain_memory_${userId}`);
      if (stored) {
        return JSON.parse(stored);
      }
    } catch (error) {
      console.error("Error loading memory from storage:", error);
    }
    return null;
  }

  private saveToStorage(userId: string, memory: UserMemory): void {
    try {
      localStorage.setItem(`brain_memory_${userId}`, JSON.stringify(memory));
    } catch (error) {
      console.error("Error saving memory to storage:", error);
    }
  }
}
