import { supabase } from "@/integrations/supabase/client";

export interface RoutingResult {
  suggestedModule: string | null;
  confidence: number;
  reasoning: string;
}

export class BrainRouter {
  private moduleCache: Map<string, any> = new Map();
  private cacheExpiry: number = 5 * 60 * 1000; // 5 minutes
  private lastCacheUpdate: number = 0;

  async classifyAndRoute(text: string, locale: string): Promise<RoutingResult> {
    try {
      // Load available modules
      const modules = await this.getActiveModules();

      // Classify intent using keywords and patterns
      const classification = this.classifyIntent(text, locale, modules);

      return classification;
    } catch (error) {
      console.error("Routing error:", error);
      return {
        suggestedModule: null,
        confidence: 0,
        reasoning: "Unable to classify",
      };
    }
  }

  private async getActiveModules(): Promise<any[]> {
    const now = Date.now();
    
    // Use cache if valid
    if (this.moduleCache.size > 0 && now - this.lastCacheUpdate < this.cacheExpiry) {
      return Array.from(this.moduleCache.values());
    }

    try {
      const { data, error } = await supabase
        .from("modules")
        .select("*")
        .eq("is_active", true)
        .order("order_index");

      if (error) throw error;

      // Update cache
      this.moduleCache.clear();
      data?.forEach(module => {
        this.moduleCache.set(module.route, module);
      });
      this.lastCacheUpdate = now;

      return data || [];
    } catch (error) {
      console.error("Error loading modules:", error);
      return [];
    }
  }

  private classifyIntent(text: string, locale: string, modules: any[]): RoutingResult {
    const lowerText = text.toLowerCase();

    // Education/Academy keywords
    if (this.matchesKeywords(lowerText, [
      "class", "study", "exam", "homework", "শ্রেণি", "পড়া", "পরীক্ষা", "হোমওয়ার্ক",
      "math", "science", "history", "গণিত", "বিজ্ঞান", "ইতিহাস"
    ])) {
      return {
        suggestedModule: "smart-academy",
        confidence: 0.85,
        reasoning: "Educational content detected",
      };
    }

    // Health/Hospital keywords
    if (this.matchesKeywords(lowerText, [
      "doctor", "appointment", "health", "medicine", "ডাক্তার", "স্বাস্থ্য", "ঔষধ",
      "sick", "pain", "fever", "অসুস্থ", "ব্যথা", "জ্বর"
    ])) {
      return {
        suggestedModule: "smart-doctor",
        confidence: 0.85,
        reasoning: "Healthcare query detected",
      };
    }

    // Design/Creative keywords
    if (this.matchesKeywords(lowerText, [
      "design", "poster", "logo", "banner", "ডিজাইন", "পোস্টার", "লোগো",
      "create", "graphic", "image", "তৈরি", "ছবি"
    ])) {
      return {
        suggestedModule: "smart-design-hub",
        confidence: 0.80,
        reasoning: "Design/creative request detected",
      };
    }

    // Marketing keywords
    if (this.matchesKeywords(lowerText, [
      "marketing", "campaign", "advertise", "মার্কেটিং", "প্রচার", "বিজ্ঞাপন",
      "promote", "social media", "প্রমোট", "সোশ্যাল মিডিয়া"
    ])) {
      return {
        suggestedModule: "smart-marketer",
        confidence: 0.80,
        reasoning: "Marketing query detected",
      };
    }

    // Shopping/Bazar keywords
    if (this.matchesKeywords(lowerText, [
      "buy", "shop", "product", "কিনতে", "দোকান", "পণ্য",
      "price", "order", "দাম", "অর্ডার"
    ])) {
      return {
        suggestedModule: "smart-bazar",
        confidence: 0.80,
        reasoning: "Shopping query detected",
      };
    }

    // Job/Career keywords
    if (this.matchesKeywords(lowerText, [
      "job", "career", "resume", "চাকরি", "ক্যারিয়ার", "জীবনবৃত্তান্ত",
      "application", "interview", "আবেদন", "ইন্টারভিউ"
    ])) {
      return {
        suggestedModule: "smart-job",
        confidence: 0.80,
        reasoning: "Career/job query detected",
      };
    }

    // Legal keywords
    if (this.matchesKeywords(lowerText, [
      "lawyer", "legal", "case", "আইনজীবী", "আইনি", "মামলা",
      "court", "law", "আদালত", "আইন"
    ])) {
      return {
        suggestedModule: "smart-lawyer",
        confidence: 0.80,
        reasoning: "Legal query detected",
      };
    }

    // News keywords
    if (this.matchesKeywords(lowerText, [
      "news", "latest", "update", "খবর", "সর্বশেষ", "আপডেট",
      "headline", "breaking", "শিরোনাম"
    ])) {
      return {
        suggestedModule: "smart-news",
        confidence: 0.75,
        reasoning: "News query detected",
      };
    }

    // School ERP keywords
    if (this.matchesKeywords(lowerText, [
      "attendance", "grade", "student", "উপস্থিতি", "গ্রেড", "ছাত্র",
      "teacher", "school", "শিক্ষক", "স্কুল"
    ])) {
      return {
        suggestedModule: "smart-school",
        confidence: 0.80,
        reasoning: "School management query detected",
      };
    }

    // Call Center keywords
    if (this.matchesKeywords(lowerText, [
      "support", "ticket", "complaint", "সাপোর্ট", "টিকেট", "অভিযোগ",
      "help", "issue", "সাহায্য", "সমস্যা"
    ])) {
      return {
        suggestedModule: "smart-call-center",
        confidence: 0.75,
        reasoning: "Support query detected",
      };
    }

    // No clear match
    return {
      suggestedModule: null,
      confidence: 0.3,
      reasoning: "General conversation",
    };
  }

  private matchesKeywords(text: string, keywords: string[]): boolean {
    return keywords.some(keyword => text.includes(keyword.toLowerCase()));
  }

  getModuleInfo(route: string): any {
    return this.moduleCache.get(route) || null;
  }
}
