import { supabase } from "@/integrations/supabase/client";

export interface ToolResult {
  success: boolean;
  data?: any;
  error?: string;
}

export class BrainTools {
  // Web search tool
  async webSearch(query: string): Promise<ToolResult> {
    try {
      const { data, error } = await supabase.functions.invoke("web-search", {
        body: { query },
      });

      if (error) throw error;

      return {
        success: true,
        data: data.results,
      };
    } catch (error: any) {
      console.error("Web search error:", error);
      return {
        success: false,
        error: error.message,
      };
    }
  }

  // Module generation tool
  async generateModule(params: { name: string; description: string; category: string }): Promise<ToolResult> {
    try {
      const { data, error } = await supabase.functions.invoke("module-generator", {
        body: params,
      });

      if (error) throw error;

      return {
        success: true,
        data: data,
      };
    } catch (error: any) {
      console.error("Module generation error:", error);
      return {
        success: false,
        error: error.message,
      };
    }
  }

  // Code analysis tool (Dev Brain)
  async analyzeCode(filePath: string, content: string): Promise<ToolResult> {
    try {
      // This would integrate with the Dev Brain
      return {
        success: true,
        data: { message: "Code analysis not yet implemented in tools" },
      };
    } catch (error: any) {
      return {
        success: false,
        error: error.message,
      };
    }
  }

  // Get module info tool
  async getModuleInfo(route: string): Promise<ToolResult> {
    try {
      const { data, error } = await supabase
        .from("modules")
        .select("*")
        .eq("route", route)
        .eq("is_active", true)
        .maybeSingle();

      if (error) throw error;

      return {
        success: true,
        data: data || null,
      };
    } catch (error: any) {
      return {
        success: false,
        error: error.message,
      };
    }
  }

  // List available tools
  getAvailableTools(): string[] {
    return [
      "web_search",
      "generate_module",
      "analyze_code",
      "get_module_info",
    ];
  }

  // Execute tool by name
  async executeTool(toolName: string, params: any): Promise<ToolResult> {
    switch (toolName) {
      case "web_search":
        return await this.webSearch(params.query);
      case "generate_module":
        return await this.generateModule(params);
      case "analyze_code":
        return await this.analyzeCode(params.filePath, params.content);
      case "get_module_info":
        return await this.getModuleInfo(params.route);
      default:
        return {
          success: false,
          error: `Unknown tool: ${toolName}`,
        };
    }
  }
}
