import { supabase } from "@/integrations/supabase/client";

export interface BrainMemory {
  shortTerm: Array<{ timestamp: number; content: string; type: string }>;
  longTerm: Array<{ id: string; summary: string; importance: number }>;
}

export interface BrainTask {
  id: string;
  description: string;
  status: "pending" | "running" | "completed" | "failed";
  subtasks: string[];
  result?: any;
  error?: string;
}

export interface BrainOperation {
  id: string;
  timestamp: number;
  type: string;
  details: string;
  success: boolean;
}

class BrainEngine {
  private memory: BrainMemory = {
    shortTerm: [],
    longTerm: [],
  };
  private operations: BrainOperation[] = [];
  private tasks: BrainTask[] = [];

  constructor() {
    this.loadMemory();
  }

  private loadMemory() {
    const stored = localStorage.getItem("brain_memory");
    if (stored) {
      this.memory = JSON.parse(stored);
    }
  }

  private saveMemory() {
    localStorage.setItem("brain_memory", JSON.stringify(this.memory));
  }

  async ask(message: string): Promise<string> {
    try {
      // Add to short-term memory
      this.updateMemory(message, "user_query");

      // Log operation
      this.logOperation("ask", `User query: ${message.substring(0, 50)}...`, true);

      // Call AI with context
      const { data, error } = await supabase.functions.invoke("smart-friend-chat", {
        body: { 
          message,
          context: this.getRecentContext(),
        },
      });

      if (error) throw error;

      const response = data.response || "I'm processing your request...";
      this.updateMemory(response, "ai_response");

      return response;
    } catch (error: any) {
      this.logOperation("ask", `Error: ${error.message}`, false);
      return "I encountered an error processing your request.";
    }
  }

  planTask(description: string): BrainTask {
    const task: BrainTask = {
      id: `task_${Date.now()}`,
      description,
      status: "pending",
      subtasks: this.generateSubtasks(description),
    };

    this.tasks.push(task);
    this.logOperation("plan_task", `Created task: ${description}`, true);
    return task;
  }

  private generateSubtasks(description: string): string[] {
    // Simple task breakdown logic
    const keywords = description.toLowerCase();
    const subtasks: string[] = [];

    if (keywords.includes("create") || keywords.includes("build")) {
      subtasks.push("Analyze requirements");
      subtasks.push("Design structure");
      subtasks.push("Implement code");
      subtasks.push("Test functionality");
    } else if (keywords.includes("fix") || keywords.includes("debug")) {
      subtasks.push("Identify issue");
      subtasks.push("Analyze root cause");
      subtasks.push("Apply fix");
      subtasks.push("Verify solution");
    } else {
      subtasks.push("Understand request");
      subtasks.push("Execute action");
      subtasks.push("Validate result");
    }

    return subtasks;
  }

  async executeTask(taskId: string): Promise<void> {
    const task = this.tasks.find((t) => t.id === taskId);
    if (!task) {
      throw new Error("Task not found");
    }

    task.status = "running";
    this.logOperation("execute_task", `Executing: ${task.description}`, true);

    try {
      // Simulate task execution
      for (const subtask of task.subtasks) {
        await new Promise((resolve) => setTimeout(resolve, 500));
        this.logOperation("subtask", subtask, true);
      }

      task.status = "completed";
      task.result = { success: true, message: "Task completed successfully" };
    } catch (error: any) {
      task.status = "failed";
      task.error = error.message;
      this.logOperation("execute_task", `Failed: ${error.message}`, false);
    }
  }

  updateMemory(content: string, type: string) {
    this.memory.shortTerm.push({
      timestamp: Date.now(),
      content,
      type,
    });

    // Keep only last 50 short-term memories
    if (this.memory.shortTerm.length > 50) {
      this.memory.shortTerm = this.memory.shortTerm.slice(-50);
    }

    this.saveMemory();
  }

  getMemory(): BrainMemory {
    return this.memory;
  }

  private getRecentContext(): string {
    return this.memory.shortTerm
      .slice(-10)
      .map((m) => `${m.type}: ${m.content}`)
      .join("\n");
  }

  private logOperation(type: string, details: string, success: boolean) {
    const operation: BrainOperation = {
      id: `op_${Date.now()}`,
      timestamp: Date.now(),
      type,
      details,
      success,
    };

    this.operations.push(operation);

    // Keep only last 100 operations
    if (this.operations.length > 100) {
      this.operations = this.operations.slice(-100);
    }
  }

  getOperations(): BrainOperation[] {
    return this.operations.slice(-20);
  }

  getTasks(): BrainTask[] {
    return this.tasks;
  }
}

export const brainEngine = new BrainEngine();
