export class DevBrainDiff {
  // Generate a unified diff between original and modified code
  generateDiff(original: string, modified: string, fileName: string): string {
    const originalLines = original.split("\n");
    const modifiedLines = modified.split("\n");

    let diff = `--- ${fileName} (original)\n`;
    diff += `+++ ${fileName} (modified)\n`;
    diff += `@@ -1,${originalLines.length} +1,${modifiedLines.length} @@\n`;

    const maxLines = Math.max(originalLines.length, modifiedLines.length);

    for (let i = 0; i < maxLines; i++) {
      const origLine = originalLines[i];
      const modLine = modifiedLines[i];

      if (origLine === modLine) {
        diff += ` ${origLine || ""}\n`;
      } else {
        if (origLine !== undefined) {
          diff += `-${origLine}\n`;
        }
        if (modLine !== undefined) {
          diff += `+${modLine}\n`;
        }
      }
    }

    return diff;
  }

  // Parse a diff string to extract changes
  parseDiff(diffString: string): {
    additions: number;
    deletions: number;
    changes: Array<{ type: "add" | "remove" | "unchanged"; line: string }>;
  } {
    const lines = diffString.split("\n");
    let additions = 0;
    let deletions = 0;
    const changes: Array<{ type: "add" | "remove" | "unchanged"; line: string }> = [];

    for (const line of lines) {
      if (line.startsWith("+") && !line.startsWith("+++")) {
        additions++;
        changes.push({ type: "add", line: line.substring(1) });
      } else if (line.startsWith("-") && !line.startsWith("---")) {
        deletions++;
        changes.push({ type: "remove", line: line.substring(1) });
      } else if (line.startsWith(" ")) {
        changes.push({ type: "unchanged", line: line.substring(1) });
      }
    }

    return { additions, deletions, changes };
  }

  // Create a human-readable summary of changes
  generateChangeSummary(diffString: string): string {
    const { additions, deletions } = this.parseDiff(diffString);
    
    const parts: string[] = [];
    if (additions > 0) parts.push(`+${additions} additions`);
    if (deletions > 0) parts.push(`-${deletions} deletions`);
    
    return parts.join(", ") || "No changes";
  }
}
