import { supabase } from "@/integrations/supabase/client";
import { DevBrainPlanner } from "./DevBrainPlanner";
import { DevBrainDiff } from "./DevBrainDiff";
import { DevBrainFileOps } from "./DevBrainFileOps";

export interface DevBrainAnalysis {
  type: "file_analysis" | "error_analysis" | "refactor_suggestion" | "upgrade_plan";
  summary: string;
  details: string;
  suggestedChanges?: string;
  affectedFiles?: string[];
}

export interface DevBrainTask {
  id: string;
  description: string;
  status: "pending" | "in_progress" | "completed" | "failed";
  steps: string[];
  result?: string;
}

class DevBrainEngine {
  private planner: DevBrainPlanner;
  private differ: DevBrainDiff;
  private fileOps: DevBrainFileOps;

  constructor() {
    this.planner = new DevBrainPlanner();
    this.differ = new DevBrainDiff();
    this.fileOps = new DevBrainFileOps();
  }

  // Analyze a file for potential issues or improvements
  async analyzeFile(filePath: string, content: string): Promise<DevBrainAnalysis> {
    try {
      await this.logAction("analyze_file", filePath, "Analyzing file for improvements");

      const { data, error } = await supabase.functions.invoke("smart-friend-chat", {
        body: {
          message: `Analyze this code file and provide suggestions for improvement:\n\nFile: ${filePath}\n\nCode:\n${content}`,
          context: "dev_brain_analysis",
        },
      });

      if (error) throw error;

      return {
        type: "file_analysis",
        summary: "Code analysis completed",
        details: data.response || "No specific suggestions at this time.",
        affectedFiles: [filePath],
      };
    } catch (error: any) {
      console.error("Error analyzing file:", error);
      throw new Error(`Failed to analyze file: ${error.message}`);
    }
  }

  // Analyze error logs
  async analyzeError(errorLog: string): Promise<DevBrainAnalysis> {
    try {
      await this.logAction("analyze_error", "", "Analyzing error log");

      const { data, error } = await supabase.functions.invoke("smart-friend-chat", {
        body: {
          message: `Analyze this error and suggest a fix:\n\n${errorLog}`,
          context: "error_analysis",
        },
      });

      if (error) throw error;

      return {
        type: "error_analysis",
        summary: "Error analysis completed",
        details: data.response || "Could not determine the cause.",
      };
    } catch (error: any) {
      console.error("Error analyzing error:", error);
      throw new Error(`Failed to analyze error: ${error.message}`);
    }
  }

  // Suggest refactoring for selected code
  async suggestRefactor(params: {
    filePath: string;
    selectedCode: string;
    goal: string;
  }): Promise<{ diff: string; newCode: string }> {
    try {
      await this.logAction("suggest_refactor", params.filePath, `Suggesting refactor: ${params.goal}`);

      const { data, error } = await supabase.functions.invoke("smart-friend-chat", {
        body: {
          message: `Refactor this code to ${params.goal}:\n\n${params.selectedCode}\n\nProvide the improved code.`,
          context: "refactor_suggestion",
        },
      });

      if (error) throw error;

      const newCode = data.response || params.selectedCode;
      const diff = this.differ.generateDiff(params.selectedCode, newCode, params.filePath);

      return { diff, newCode };
    } catch (error: any) {
      console.error("Error suggesting refactor:", error);
      throw new Error(`Failed to suggest refactor: ${error.message}`);
    }
  }

  // Generate upgrade plan for the entire project
  async generateUpgradePlan(): Promise<DevBrainAnalysis> {
    try {
      await this.logAction("generate_upgrade_plan", "", "Generating project upgrade plan");

      const { data, error } = await supabase.functions.invoke("smart-friend-chat", {
        body: {
          message: "Generate a comprehensive upgrade plan for this project. Consider: dependencies, architecture, performance, security, and best practices.",
          context: "upgrade_plan",
        },
      });

      if (error) throw error;

      return {
        type: "upgrade_plan",
        summary: "Upgrade plan generated",
        details: data.response || "No upgrades needed at this time.",
      };
    } catch (error: any) {
      console.error("Error generating upgrade plan:", error);
      throw new Error(`Failed to generate upgrade plan: ${error.message}`);
    }
  }

  // Apply code changes after owner approval
  async applyChanges(filePath: string, newCode: string, diff: string): Promise<void> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("User not authenticated");

      await this.logAction("apply_changes", filePath, "Applying code changes", diff);

      // In a real implementation, this would write to the actual file system
      // For now, we just log the action
      console.log("Changes would be applied to:", filePath);

      await this.updateLogStatus("applied");
    } catch (error: any) {
      console.error("Error applying changes:", error);
      throw new Error(`Failed to apply changes: ${error.message}`);
    }
  }

  // Log an action to the database
  private async logAction(
    actionType: string,
    filePath: string,
    description: string,
    diffContent?: string
  ): Promise<void> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      await supabase.from("dev_brain_logs").insert({
        user_id: user.id,
        action_type: actionType,
        file_path: filePath || null,
        description,
        diff_content: diffContent || null,
        status: "pending",
      });
    } catch (error) {
      console.error("Error logging action:", error);
    }
  }

  // Update log status
  private async updateLogStatus(status: string): Promise<void> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return;

      // Get most recent log
      const { data: logs } = await supabase
        .from("dev_brain_logs")
        .select("*")
        .eq("user_id", user.id)
        .order("created_at", { ascending: false })
        .limit(1);

      if (logs && logs.length > 0) {
        await supabase
          .from("dev_brain_logs")
          .update({ 
            status,
            applied_at: status === "applied" ? new Date().toISOString() : null
          })
          .eq("id", logs[0].id);
      }
    } catch (error) {
      console.error("Error updating log status:", error);
    }
  }
}

export const devBrainEngine = new DevBrainEngine();
