import { supabase } from "@/integrations/supabase/client";

export interface FileInfo {
  path: string;
  name: string;
  type: "file" | "folder";
  content?: string;
  language?: string;
}

export class DevBrainFileOps {
  // Get project file structure
  async getProjectFiles(): Promise<FileInfo[]> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("User not authenticated");

      const { data, error } = await supabase
        .from("project_files")
        .select("*")
        .eq("user_id", user.id)
        .order("file_path");

      if (error) throw error;

      return (data || []).map(file => ({
        path: file.file_path,
        name: file.file_name,
        type: file.file_type as "file" | "folder",
        content: file.content || undefined,
        language: this.detectLanguage(file.file_name),
      }));
    } catch (error) {
      console.error("Error fetching project files:", error);
      return [];
    }
  }

  // Read file content
  async readFile(filePath: string): Promise<string> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("User not authenticated");

      const { data, error } = await supabase
        .from("project_files")
        .select("content")
        .eq("user_id", user.id)
        .eq("file_path", filePath)
        .single();

      if (error) throw error;
      return data?.content || "";
    } catch (error) {
      console.error("Error reading file:", error);
      throw error;
    }
  }

  // Write file content (simulated - in real app would need proper file system access)
  async writeFile(filePath: string, content: string): Promise<void> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("User not authenticated");

      // In a real implementation, this would update the actual file
      // For now, we just update the database record
      const { error } = await supabase
        .from("project_files")
        .update({ content })
        .eq("user_id", user.id)
        .eq("file_path", filePath);

      if (error) throw error;
    } catch (error) {
      console.error("Error writing file:", error);
      throw error;
    }
  }

  // Detect programming language from file extension
  private detectLanguage(fileName: string): string {
    const ext = fileName.split(".").pop()?.toLowerCase();
    const languageMap: Record<string, string> = {
      ts: "typescript",
      tsx: "typescript",
      js: "javascript",
      jsx: "javascript",
      py: "python",
      java: "java",
      cpp: "cpp",
      c: "c",
      cs: "csharp",
      go: "go",
      rs: "rust",
      php: "php",
      rb: "ruby",
      swift: "swift",
      kt: "kotlin",
      dart: "dart",
      css: "css",
      scss: "scss",
      html: "html",
      json: "json",
      xml: "xml",
      yaml: "yaml",
      yml: "yaml",
      md: "markdown",
      sql: "sql",
    };

    return languageMap[ext || ""] || "plaintext";
  }

  // Get file statistics
  async getFileStats(filePath: string): Promise<{
    lines: number;
    size: number;
    language: string;
  }> {
    try {
      const content = await this.readFile(filePath);
      const lines = content.split("\n").length;
      const size = new Blob([content]).size;
      const language = this.detectLanguage(filePath);

      return { lines, size, language };
    } catch (error) {
      console.error("Error getting file stats:", error);
      return { lines: 0, size: 0, language: "unknown" };
    }
  }
}
