import { supabase } from "@/integrations/supabase/client";

export interface DevBrainLog {
  id: string;
  created_at: string;
  action_type: string;
  file_path: string | null;
  description: string;
  diff_content: string | null;
  status: string;
  applied_at: string | null;
}

export class DevBrainHistory {
  // Get recent logs
  async getRecentLogs(limit: number = 20): Promise<DevBrainLog[]> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return [];

      const { data, error } = await supabase
        .from("dev_brain_logs")
        .select("*")
        .eq("user_id", user.id)
        .order("created_at", { ascending: false })
        .limit(limit);

      if (error) throw error;
      return data || [];
    } catch (error) {
      console.error("Error fetching logs:", error);
      return [];
    }
  }

  // Get logs by file
  async getLogsByFile(filePath: string): Promise<DevBrainLog[]> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return [];

      const { data, error } = await supabase
        .from("dev_brain_logs")
        .select("*")
        .eq("user_id", user.id)
        .eq("file_path", filePath)
        .order("created_at", { ascending: false });

      if (error) throw error;
      return data || [];
    } catch (error) {
      console.error("Error fetching file logs:", error);
      return [];
    }
  }

  // Get logs by action type
  async getLogsByType(actionType: string): Promise<DevBrainLog[]> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return [];

      const { data, error } = await supabase
        .from("dev_brain_logs")
        .select("*")
        .eq("user_id", user.id)
        .eq("action_type", actionType)
        .order("created_at", { ascending: false });

      if (error) throw error;
      return data || [];
    } catch (error) {
      console.error("Error fetching logs by type:", error);
      return [];
    }
  }

  // Get statistics
  async getStatistics(): Promise<{
    totalActions: number;
    appliedChanges: number;
    pendingChanges: number;
    actionTypes: Record<string, number>;
  }> {
    try {
      const logs = await this.getRecentLogs(1000);
      
      const totalActions = logs.length;
      const appliedChanges = logs.filter(log => log.status === "applied").length;
      const pendingChanges = logs.filter(log => log.status === "pending").length;
      
      const actionTypes: Record<string, number> = {};
      logs.forEach(log => {
        actionTypes[log.action_type] = (actionTypes[log.action_type] || 0) + 1;
      });

      return {
        totalActions,
        appliedChanges,
        pendingChanges,
        actionTypes,
      };
    } catch (error) {
      console.error("Error getting statistics:", error);
      return {
        totalActions: 0,
        appliedChanges: 0,
        pendingChanges: 0,
        actionTypes: {},
      };
    }
  }

  // Delete old logs (cleanup)
  async deleteOldLogs(daysOld: number = 30): Promise<number> {
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) return 0;

      const cutoffDate = new Date();
      cutoffDate.setDate(cutoffDate.getDate() - daysOld);

      const { error } = await supabase
        .from("dev_brain_logs")
        .delete()
        .eq("user_id", user.id)
        .lt("created_at", cutoffDate.toISOString());

      if (error) throw error;
      return 0; // Deletion successful
    } catch (error) {
      console.error("Error deleting old logs:", error);
      return 0;
    }
  }
}
