export interface TaskStep {
  id: string;
  description: string;
  completed: boolean;
}

export interface DevPlan {
  id: string;
  title: string;
  description: string;
  steps: TaskStep[];
  estimatedTime?: string;
}

export class DevBrainPlanner {
  // Break down a complex task into smaller steps
  planTask(taskDescription: string): DevPlan {
    const steps: TaskStep[] = [];
    
    // Analyze the task type and create appropriate steps
    if (taskDescription.toLowerCase().includes("refactor")) {
      steps.push(
        { id: "1", description: "Analyze current code structure", completed: false },
        { id: "2", description: "Identify improvement areas", completed: false },
        { id: "3", description: "Generate refactored code", completed: false },
        { id: "4", description: "Create diff for review", completed: false },
        { id: "5", description: "Apply changes", completed: false }
      );
    } else if (taskDescription.toLowerCase().includes("error") || taskDescription.toLowerCase().includes("bug")) {
      steps.push(
        { id: "1", description: "Analyze error logs", completed: false },
        { id: "2", description: "Identify root cause", completed: false },
        { id: "3", description: "Generate fix", completed: false },
        { id: "4", description: "Test solution", completed: false },
        { id: "5", description: "Apply fix", completed: false }
      );
    } else if (taskDescription.toLowerCase().includes("feature") || taskDescription.toLowerCase().includes("add")) {
      steps.push(
        { id: "1", description: "Understand requirements", completed: false },
        { id: "2", description: "Design architecture", completed: false },
        { id: "3", description: "Generate code", completed: false },
        { id: "4", description: "Integrate with existing code", completed: false },
        { id: "5", description: "Test implementation", completed: false }
      );
    } else {
      steps.push(
        { id: "1", description: "Analyze request", completed: false },
        { id: "2", description: "Plan approach", completed: false },
        { id: "3", description: "Execute task", completed: false },
        { id: "4", description: "Verify results", completed: false }
      );
    }

    return {
      id: `plan_${Date.now()}`,
      title: taskDescription,
      description: `Development plan for: ${taskDescription}`,
      steps,
      estimatedTime: `${steps.length * 5} minutes`,
    };
  }

  // Update step completion status
  updateStepStatus(plan: DevPlan, stepId: string, completed: boolean): DevPlan {
    return {
      ...plan,
      steps: plan.steps.map(step =>
        step.id === stepId ? { ...step, completed } : step
      ),
    };
  }

  // Check if plan is complete
  isPlanComplete(plan: DevPlan): boolean {
    return plan.steps.every(step => step.completed);
  }

  // Get next pending step
  getNextStep(plan: DevPlan): TaskStep | null {
    return plan.steps.find(step => !step.completed) || null;
  }
}
