import JSZip from "jszip";

export interface ProjectFile {
  path: string;
  content: string;
}

export class ZipBuilder {
  async buildProductionZip(
    projectName: string = "smart-friend",
    files: ProjectFile[] = []
  ): Promise<Blob> {
    const zip = new JSZip();

    // Add README_DEPLOY.md
    const readme = this.generateDeploymentReadme();
    zip.file("README_DEPLOY.md", readme);

    // Add .env.example
    const envExample = this.generateEnvExample();
    zip.file(".env.example", envExample);

    // Add package.json
    const packageJson = this.generatePackageJson(projectName);
    zip.file("package.json", JSON.stringify(packageJson, null, 2));

    // Add project files if provided
    files.forEach((file) => {
      zip.file(file.path, file.content);
    });

    // Generate the ZIP
    const blob = await zip.generateAsync({ type: "blob" });
    return blob;
  }

  private generateDeploymentReadme(): string {
    return `# Smart Friend - Deployment Guide

## Owner: Engineer S M Nazrul Islam Tasnim (HQ)

## Local Development

### 1. Install Dependencies
\`\`\`bash
npm install
\`\`\`

### 2. Configure Environment
\`\`\`bash
cp .env.example .env
\`\`\`

Edit \`.env\` and add your API keys:
- VITE_SUPABASE_URL
- VITE_SUPABASE_ANON_KEY
- VITE_OPENAI_API_KEY (optional)

### 3. Run Development Server
\`\`\`bash
npm run dev
\`\`\`

## Production Build

### 1. Build the Project
\`\`\`bash
npm run build
\`\`\`

This creates a \`/dist\` folder with production-ready files.

## Hosting on cPanel (Shared Hosting)

### Step 1: Upload Files
1. Build the project locally: \`npm run build\`
2. Access your cPanel File Manager
3. Navigate to \`public_html/smartfriend\` (or your desired folder)
4. Upload all files from the \`/dist\` folder

### Step 2: Configure Environment Variables
1. Create a \`.env\` file in the root directory
2. Add your production API keys
3. Ensure the file is not publicly accessible

### Step 3: Set Base URL
In your Vite config, ensure:
\`\`\`javascript
base: '/smartfriend/' // if hosted in a subfolder
\`\`\`

## Hosting on Netlify/Vercel

### Netlify
1. Connect your GitHub repository
2. Build command: \`npm run build\`
3. Publish directory: \`dist\`
4. Add environment variables in Netlify dashboard

### Vercel
1. Import your GitHub repository
2. Framework: Vite
3. Build command: \`npm run build\`
4. Output directory: \`dist\`
5. Add environment variables in Vercel dashboard

## Environment Variables

Required:
- \`VITE_SUPABASE_URL\` - Your Supabase project URL
- \`VITE_SUPABASE_ANON_KEY\` - Your Supabase anon key
- \`VITE_APP_BASE_URL\` - Your production domain

Optional:
- \`VITE_OPENAI_API_KEY\` - For AI features
- \`VITE_APP_ENV\` - Set to 'production'

## Changing API Keys

### Supabase
1. Go to your Supabase project dashboard
2. Settings → API
3. Copy the new keys
4. Update your \`.env\` file

### OpenAI
1. Go to platform.openai.com
2. API Keys → Create new key
3. Update your \`.env\` file

## Troubleshooting

### Build Errors
- Clear node_modules: \`rm -rf node_modules && npm install\`
- Clear cache: \`npm run build -- --force\`

### Routing Issues
- Ensure your hosting supports SPA routing
- Add a \`_redirects\` file for Netlify
- Add \`vercel.json\` for Vercel

## Support
For issues or questions, contact: engineer.tasnim@smartseba.com
`;
  }

  private generateEnvExample(): string {
    return `# Smart Friend Environment Variables

# Supabase Configuration
VITE_SUPABASE_URL=your_supabase_url_here
VITE_SUPABASE_ANON_KEY=your_supabase_anon_key_here

# OpenAI (Optional)
VITE_OPENAI_API_KEY=your_openai_api_key_here

# App Configuration
VITE_APP_BASE_URL=https://yourdomain.com
VITE_APP_ENV=production
`;
  }

  private generatePackageJson(projectName: string): any {
    return {
      name: projectName,
      version: "1.0.0",
      description: "Smart Friend - AI-powered platform by Engineer Tasnim",
      author: "Engineer S M Nazrul Islam Tasnim",
      scripts: {
        dev: "vite",
        build: "vite build",
        preview: "vite preview",
        lint: "eslint .",
      },
      dependencies: {
        react: "^18.3.1",
        "react-dom": "^18.3.1",
        "react-router-dom": "^6.30.1",
        "@supabase/supabase-js": "^2.81.1",
        "@tanstack/react-query": "^5.83.0",
        "lucide-react": "latest",
        jszip: "^3.10.1",
      },
      devDependencies: {
        vite: "^5.0.0",
        "@vitejs/plugin-react": "^4.2.0",
        typescript: "^5.3.0",
        tailwindcss: "^3.4.0",
      },
    };
  }

  downloadZip(blob: Blob, filename: string = "smart-friend-production.zip"): void {
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  }
}

export const zipBuilder = new ZipBuilder();
