// Phase 3: Backend Code Generator

import { ModuleBlueprint, BlueprintEntity } from '@/types/moduleBlueprint';

export class BackendGenerator {
  /**
   * Generate backend code based on blueprint
   */
  generateCode(blueprint: ModuleBlueprint): Record<string, string> {
    console.log('🔧 Generating backend code...', blueprint.backendStack);

    switch (blueprint.backendStack) {
      case 'laravel':
        return this.generateLaravel(blueprint);
      case 'dotnet':
        return this.generateDotNet(blueprint);
      case 'express':
        return this.generateExpress(blueprint);
      default:
        throw new Error(`Unsupported backend stack: ${blueprint.backendStack}`);
    }
  }

  private generateLaravel(blueprint: ModuleBlueprint): Record<string, string> {
    const files: Record<string, string> = {};

    // Generate migrations
    blueprint.entities.forEach(entity => {
      const migrationName = `create_${entity.name.toLowerCase()}s_table`;
      files[`database/migrations/${migrationName}.php`] = this.generateLaravelMigration(entity);
    });

    // Generate models
    blueprint.entities.forEach(entity => {
      files[`app/Models/${entity.name}.php`] = this.generateLaravelModel(entity, blueprint);
    });

    // Generate controllers
    blueprint.entities.forEach(entity => {
      files[`app/Http/Controllers/${entity.name}Controller.php`] = this.generateLaravelController(entity, blueprint);
    });

    // Generate routes
    files['routes/api.php'] = this.generateLaravelRoutes(blueprint);

    // Generate requests
    blueprint.entities.forEach(entity => {
      files[`app/Http/Requests/Store${entity.name}Request.php`] = this.generateLaravelRequest(entity, 'Store');
      files[`app/Http/Requests/Update${entity.name}Request.php`] = this.generateLaravelRequest(entity, 'Update');
    });

    return files;
  }

  private generateLaravelMigration(entity: BlueprintEntity): string {
    const tableName = entity.name.toLowerCase() + 's';
    const fields = entity.fields
      .filter(f => f.name !== 'id')
      .map(field => this.laravelFieldToMigration(field))
      .join('\n            ');

    return `<?php

use Illuminate\\Database\\Migrations\\Migration;
use Illuminate\\Database\\Schema\\Blueprint;
use Illuminate\\Support\\Facades\\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('${tableName}', function (Blueprint $table) {
            $table->id();
            ${fields}
            ${entity.hasTimestamps ? '$table->timestamps();' : ''}
            ${entity.hasSoftDelete ? '$table->softDeletes();' : ''}
            ${entity.hasOwnership ? '$table->foreignId(\'user_id\')->constrained()->onDelete(\'cascade\');' : ''}
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('${tableName}');
    }
};
`;
  }

  private laravelFieldToMigration(field: any): string {
    const typeMap: Record<string, string> = {
      string: 'string',
      number: 'integer',
      boolean: 'boolean',
      date: 'timestamp',
      text: 'text',
      json: 'json',
    };

    const type = typeMap[field.type] || 'string';
    let migration = `$table->${type}('${field.name}')`;

    if (!field.required) migration += '->nullable()';
    if (field.defaultValue !== undefined) {
      const value = typeof field.defaultValue === 'string' ? `'${field.defaultValue}'` : field.defaultValue;
      migration += `->default(${value})`;
    }
    if (field.isUnique) migration += '->unique()';

    return migration + ';';
  }

  private generateLaravelModel(entity: BlueprintEntity, blueprint: ModuleBlueprint): string {
    const fillable = entity.fields.filter(f => f.name !== 'id').map(f => `'${f.name}'`).join(', ');
    
    return `<?php

namespace App\\Models;

use Illuminate\\Database\\Eloquent\\Factories\\HasFactory;
use Illuminate\\Database\\Eloquent\\Model;
${entity.hasSoftDelete ? 'use Illuminate\\Database\\Eloquent\\SoftDeletes;' : ''}

class ${entity.name} extends Model
{
    use HasFactory${entity.hasSoftDelete ? ', SoftDeletes' : ''};

    protected $fillable = [${fillable}];

    protected $casts = [
        ${entity.fields.filter(f => f.type === 'json').map(f => `'${f.name}' => 'array'`).join(',\n        ')},
        ${entity.fields.filter(f => f.type === 'boolean').map(f => `'${f.name}' => 'boolean'`).join(',\n        ')},
        ${entity.fields.filter(f => f.type === 'date').map(f => `'${f.name}' => 'datetime'`).join(',\n        ')},
    ];

    ${entity.hasOwnership ? `
    public function user()
    {
        return $this->belongsTo(User::class);
    }` : ''}
}
`;
  }

  private generateLaravelController(entity: BlueprintEntity, blueprint: ModuleBlueprint): string {
    const modelName = entity.name;
    const tableName = entity.name.toLowerCase() + 's';

    return `<?php

namespace App\\Http\\Controllers;

use App\\Models\\${modelName};
use App\\Http\\Requests\\Store${modelName}Request;
use App\\Http\\Requests\\Update${modelName}Request;
use Illuminate\\Http\\JsonResponse;

class ${modelName}Controller extends Controller
{
    public function index(): JsonResponse
    {
        $items = ${modelName}::query()
            ${entity.hasOwnership ? '->where(\'user_id\', auth()->id())' : ''}
            ->paginate(15);

        return response()->json($items);
    }

    public function store(Store${modelName}Request $request): JsonResponse
    {
        $data = $request->validated();
        ${entity.hasOwnership ? '$data[\'user_id\'] = auth()->id();' : ''}

        $item = ${modelName}::create($data);

        return response()->json($item, 201);
    }

    public function show(${modelName} $${tableName.slice(0, -1)}): JsonResponse
    {
        ${entity.hasOwnership ? `
        if ($${tableName.slice(0, -1)}->user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }` : ''}

        return response()->json($${tableName.slice(0, -1)});
    }

    public function update(Update${modelName}Request $request, ${modelName} $${tableName.slice(0, -1)}): JsonResponse
    {
        ${entity.hasOwnership ? `
        if ($${tableName.slice(0, -1)}->user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }` : ''}

        $${tableName.slice(0, -1)}->update($request->validated());

        return response()->json($${tableName.slice(0, -1)});
    }

    public function destroy(${modelName} $${tableName.slice(0, -1)}): JsonResponse
    {
        ${entity.hasOwnership ? `
        if ($${tableName.slice(0, -1)}->user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }` : ''}

        $${tableName.slice(0, -1)}->delete();

        return response()->json(null, 204);
    }
}
`;
  }

  private generateLaravelRoutes(blueprint: ModuleBlueprint): string {
    const routes = blueprint.entities.map(entity => {
      const controller = `${entity.name}Controller`;
      const path = entity.name.toLowerCase() + 's';
      return `Route::apiResource('${path}', ${controller}::class);`;
    }).join('\n');

    return `<?php

use Illuminate\\Support\\Facades\\Route;
use App\\Http\\Controllers\\*;

Route::middleware(['auth:sanctum'])->group(function () {
    ${routes}
});
`;
  }

  private generateLaravelRequest(entity: BlueprintEntity, type: 'Store' | 'Update'): string {
    const rules = entity.fields
      .filter(f => f.name !== 'id')
      .map(field => {
        const required = type === 'Store' && field.required ? 'required' : 'sometimes';
        const typeRule = field.type === 'string' ? 'string' : field.type === 'number' ? 'integer' : field.type;
        return `'${field.name}' => ['${required}', '${typeRule}']`;
      })
      .join(',\n            ');

    return `<?php

namespace App\\Http\\Requests;

use Illuminate\\Foundation\\Http\\FormRequest;

class ${type}${entity.name}Request extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            ${rules}
        ];
    }
}
`;
  }

  private generateDotNet(blueprint: ModuleBlueprint): Record<string, string> {
    // .NET generation would be implemented here
    return {
      'README.md': '# .NET backend generation coming soon',
    };
  }

  private generateExpress(blueprint: ModuleBlueprint): Record<string, string> {
    // Express generation would be implemented here
    return {
      'README.md': '# Express backend generation coming soon',
    };
  }
}

export const backendGenerator = new BackendGenerator();
