// Phase 3: Builder Runner - Orchestrates the entire build process

import { ModuleBlueprint } from '@/types/moduleBlueprint';
import { naturalToBlueprint } from './NaturalToBlueprint';
import { backendGenerator } from './BackendGenerator';
import { frontendGenerator } from './FrontendGenerator';
import { docsGenerator } from './DocsGenerator';

export interface BuildResult {
  success: boolean;
  blueprint: ModuleBlueprint;
  files: Record<string, string>;
  error?: string;
}

export class BuilderRunner {
  /**
   * Run the complete build process from natural description to generated code
   */
  async buildModule(description: string, locale: string = 'en'): Promise<BuildResult> {
    console.log('🚀 Starting module build process...');

    try {
      // Step 1: Parse natural language to blueprint
      console.log('📋 Step 1: Creating blueprint...');
      const blueprint = await naturalToBlueprint.parseDescription(description, locale);

      // Step 2: Generate backend code
      console.log('🔧 Step 2: Generating backend code...');
      const backendFiles = backendGenerator.generateCode(blueprint);

      // Step 3: Generate frontend code
      console.log('🎨 Step 3: Generating frontend code...');
      const frontendFiles = frontendGenerator.generateCode(blueprint);

      // Step 4: Generate documentation
      console.log('📚 Step 4: Generating documentation...');
      const docFiles = docsGenerator.generateDocs(blueprint);

      // Combine all files
      const allFiles = {
        ...this.prefixKeys(backendFiles, 'backend/'),
        ...this.prefixKeys(frontendFiles, 'frontend/'),
        ...this.prefixKeys(docFiles, 'docs/'),
      };

      console.log('✅ Module build complete!');
      console.log(`Generated ${Object.keys(allFiles).length} files`);

      return {
        success: true,
        blueprint: {
          ...blueprint,
          status: 'generated',
          generatedFiles: allFiles,
        },
        files: allFiles,
      };
    } catch (error) {
      console.error('❌ Module build failed:', error);
      
      return {
        success: false,
        blueprint: {
          name: 'Failed Build',
          slug: 'failed',
          description,
          entities: [],
          relationships: [],
          uiPages: [],
          apiRoutes: [],
          backendStack: 'laravel',
          frontendStack: 'react',
          permissions: [],
          menuItems: [],
          complexity: 'simple',
          status: 'failed',
          version: '0.0.0',
        },
        files: {},
        error: error instanceof Error ? error.message : 'Unknown error',
      };
    }
  }

  /**
   * Build a complete new project with multiple modules
   */
  async buildProject(
    name: string,
    description: string,
    modules: string[],
    options: {
      backendStack: 'laravel' | 'dotnet' | 'express';
      frontendStack: 'react' | 'next' | 'vue';
      database: 'mysql' | 'postgresql' | 'sqlite';
      authentication: boolean;
    }
  ): Promise<BuildResult> {
    console.log('🏗️ Building complete project:', name);

    try {
      // Build each module
      const moduleResults = await Promise.all(
        modules.map(desc => this.buildModule(desc))
      );

      // Combine all module files
      const allFiles: Record<string, string> = {};
      const blueprints: ModuleBlueprint[] = [];

      moduleResults.forEach((result, index) => {
        if (result.success) {
          blueprints.push(result.blueprint);
          Object.entries(result.files).forEach(([path, content]) => {
            allFiles[`modules/${result.blueprint.slug}/${path}`] = content;
          });
        }
      });

      // Generate project-level files
      allFiles['README.md'] = this.generateProjectReadme(name, description, blueprints);
      allFiles['docker-compose.yml'] = this.generateDockerCompose(options);
      allFiles['.env.example'] = this.generateEnvExample(options);

      console.log('✅ Project build complete!');

      return {
        success: true,
        blueprint: {
          name,
          slug: name.toLowerCase().replace(/\s+/g, '-'),
          description,
          entities: [],
          relationships: [],
          uiPages: [],
          apiRoutes: [],
          backendStack: options.backendStack,
          frontendStack: options.frontendStack,
          permissions: [],
          menuItems: [],
          complexity: blueprints.length > 3 ? 'complex' : 'medium',
          status: 'generated',
          version: '1.0.0',
        },
        files: allFiles,
      };
    } catch (error) {
      console.error('❌ Project build failed:', error);
      
      return {
        success: false,
        blueprint: {
          name,
          slug: 'failed',
          description,
          entities: [],
          relationships: [],
          uiPages: [],
          apiRoutes: [],
          backendStack: options.backendStack,
          frontendStack: options.frontendStack,
          permissions: [],
          menuItems: [],
          complexity: 'simple',
          status: 'failed',
          version: '0.0.0',
        },
        files: {},
        error: error instanceof Error ? error.message : 'Unknown error',
      };
    }
  }

  private prefixKeys(obj: Record<string, string>, prefix: string): Record<string, string> {
    const result: Record<string, string> = {};
    Object.entries(obj).forEach(([key, value]) => {
      result[prefix + key] = value;
    });
    return result;
  }

  private generateProjectReadme(name: string, description: string, modules: ModuleBlueprint[]): string {
    return `# ${name}

${description}

## Modules

${modules.map(m => `- **${m.name}**: ${m.description}`).join('\n')}

## Getting Started

See individual module READMEs for setup instructions.

---

Generated by Smart Friend Ultra
`;
  }

  private generateDockerCompose(options: any): string {
    return `version: '3.8'

services:
  database:
    image: ${options.database === 'postgresql' ? 'postgres:15' : 'mysql:8'}
    environment:
      ${options.database === 'postgresql' ? 'POSTGRES_PASSWORD' : 'MYSQL_ROOT_PASSWORD'}: password
    ports:
      - "${options.database === 'postgresql' ? '5432:5432' : '3306:3306'}"
    volumes:
      - db_data:/var/lib/${options.database === 'postgresql' ? 'postgresql' : 'mysql'}/data

volumes:
  db_data:
`;
  }

  private generateEnvExample(options: any): string {
    return `# Database
DB_CONNECTION=${options.database}
DB_HOST=localhost
DB_PORT=${options.database === 'postgresql' ? '5432' : '3306'}
DB_DATABASE=smart_friend
DB_USERNAME=root
DB_PASSWORD=password

# API
API_URL=http://localhost:8000

# Authentication
${options.authentication ? 'JWT_SECRET=your-secret-key' : '# Authentication disabled'}
`;
  }
}

export const builderRunner = new BuilderRunner();
