// Phase 3: Documentation Generator

import { ModuleBlueprint } from '@/types/moduleBlueprint';

export class DocsGenerator {
  /**
   * Generate comprehensive documentation for the module
   */
  generateDocs(blueprint: ModuleBlueprint): Record<string, string> {
    const files: Record<string, string> = {};

    files['README.md'] = this.generateReadme(blueprint);
    files['API.md'] = this.generateAPIDoc(blueprint);
    files['SETUP.md'] = this.generateSetupDoc(blueprint);

    return files;
  }

  private generateReadme(blueprint: ModuleBlueprint): string {
    return `# ${blueprint.name}

${blueprint.description}

## Overview

This module was auto-generated by Smart Friend Ultra and includes:

- **Backend**: ${blueprint.backendStack}
- **Frontend**: ${blueprint.frontendStack}
- **Complexity**: ${blueprint.complexity}
- **Version**: ${blueprint.version}

## Entities

${blueprint.entities.map(entity => `
### ${entity.displayName}

${entity.fields.map(f => `- **${f.name}** (${f.type})${f.required ? ' *required*' : ''}`).join('\n')}
`).join('\n')}

## Features

${blueprint.uiPages.map(page => `- ${page.title} (${page.type})`).join('\n')}

## Permissions

${blueprint.permissions.map(p => `- ${p}`).join('\n')}

## Getting Started

See [SETUP.md](./SETUP.md) for installation instructions.

## API Documentation

See [API.md](./API.md) for API endpoints and usage.

---

Generated by Smart Friend Ultra v${blueprint.version}
`;
  }

  private generateAPIDoc(blueprint: ModuleBlueprint): string {
    return `# API Documentation

## Endpoints

${blueprint.apiRoutes.map(route => `
### ${route.method} ${route.path}

- **Action**: ${route.action}
- **Auth**: ${route.auth ? 'Required' : 'Public'}
- **Permissions**: ${route.permissions.join(', ')}

**Example Request:**
\`\`\`bash
curl -X ${route.method} \\
  http://localhost:8000/api${route.path} \\
  -H "Authorization: Bearer YOUR_TOKEN" \\
  -H "Content-Type: application/json"
\`\`\`
`).join('\n')}

## Authentication

All authenticated endpoints require a Bearer token in the Authorization header.

## Error Responses

- **401**: Unauthorized
- **403**: Forbidden
- **404**: Not Found
- **422**: Validation Error
- **500**: Server Error
`;
  }

  private generateSetupDoc(blueprint: ModuleBlueprint): string {
    const backendSetup = blueprint.backendStack === 'laravel' 
      ? `
## Backend Setup (Laravel)

1. Install dependencies:
\`\`\`bash
composer install
\`\`\`

2. Configure environment:
\`\`\`bash
cp .env.example .env
php artisan key:generate
\`\`\`

3. Run migrations:
\`\`\`bash
php artisan migrate
\`\`\`

4. Start server:
\`\`\`bash
php artisan serve
\`\`\`
`
      : '## Backend Setup\n\nComing soon';

    const frontendSetup = blueprint.frontendStack === 'react'
      ? `
## Frontend Setup (React)

1. Install dependencies:
\`\`\`bash
npm install
\`\`\`

2. Configure environment:
\`\`\`bash
cp .env.example .env
\`\`\`

3. Start development server:
\`\`\`bash
npm run dev
\`\`\`
`
      : '## Frontend Setup\n\nComing soon';

    return `# Setup Guide

## Prerequisites

- Node.js 18+
- ${blueprint.backendStack === 'laravel' ? 'PHP 8.2+, Composer' : 'Backend runtime'}
- Database (MySQL/PostgreSQL)

${backendSetup}

${frontendSetup}

## Environment Variables

### Backend
\`\`\`
APP_NAME=${blueprint.name}
APP_ENV=local
APP_DEBUG=true
DB_CONNECTION=mysql
DB_HOST=127.0.0.1
DB_PORT=3306
DB_DATABASE=${blueprint.slug}
DB_USERNAME=root
DB_PASSWORD=
\`\`\`

### Frontend
\`\`\`
VITE_API_URL=http://localhost:8000/api
VITE_SUPABASE_URL=your_supabase_url
VITE_SUPABASE_ANON_KEY=your_supabase_key
\`\`\`

## Deployment

See deployment documentation for production setup instructions.
`;
  }
}

export const docsGenerator = new DocsGenerator();
