// Phase 3: Frontend Code Generator

import { ModuleBlueprint, BlueprintEntity, BlueprintPage } from '@/types/moduleBlueprint';

export class FrontendGenerator {
  /**
   * Generate frontend code based on blueprint
   */
  generateCode(blueprint: ModuleBlueprint): Record<string, string> {
    console.log('🎨 Generating frontend code...', blueprint.frontendStack);

    switch (blueprint.frontendStack) {
      case 'react':
        return this.generateReact(blueprint);
      case 'next':
        return this.generateNext(blueprint);
      case 'vue':
        return this.generateVue(blueprint);
      default:
        throw new Error(`Unsupported frontend stack: ${blueprint.frontendStack}`);
    }
  }

  private generateReact(blueprint: ModuleBlueprint): Record<string, string> {
    const files: Record<string, string> = {};

    // Generate pages
    blueprint.entities.forEach(entity => {
      files[`src/pages/${entity.name}List.tsx`] = this.generateReactListPage(entity, blueprint);
      files[`src/pages/${entity.name}Form.tsx`] = this.generateReactFormPage(entity, blueprint);
      files[`src/pages/${entity.name}View.tsx`] = this.generateReactViewPage(entity, blueprint);
    });

    // Generate hooks
    blueprint.entities.forEach(entity => {
      files[`src/hooks/use${entity.name}.tsx`] = this.generateReactHook(entity, blueprint);
    });

    // Generate types
    files['src/types/module.ts'] = this.generateTypeScript(blueprint);

    return files;
  }

  private generateReactListPage(entity: BlueprintEntity, blueprint: ModuleBlueprint): string {
    const hookName = `use${entity.name}`;
    const pluralName = entity.name.toLowerCase() + 's';

    return `import { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { ${hookName} } from '@/hooks/${hookName}';
import { Button } from '@/components/ui/button';
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Plus, Edit, Trash2 } from 'lucide-react';

const ${entity.name}List = () => {
  const navigate = useNavigate();
  const { ${pluralName}, isLoading, delete${entity.name} } = ${hookName}();

  const handleDelete = async (id: string) => {
    if (confirm('Are you sure you want to delete this item?')) {
      await delete${entity.name}.mutateAsync(id);
    }
  };

  if (isLoading) {
    return <div>Loading...</div>;
  }

  return (
    <div className="container mx-auto p-6">
      <div className="flex justify-between items-center mb-6">
        <h1 className="text-3xl font-bold">${entity.displayName} List</h1>
        <Button onClick={() => navigate('/${blueprint.slug}/${entity.name.toLowerCase()}/new')}>
          <Plus className="h-4 w-4 mr-2" />
          Add New
        </Button>
      </div>

      <Table>
        <TableHeader>
          <TableRow>
            ${entity.fields.slice(0, 4).map(f => `<TableHead>${f.name}</TableHead>`).join('\n            ')}
            <TableHead>Actions</TableHead>
          </TableRow>
        </TableHeader>
        <TableBody>
          {${pluralName}?.map((item: any) => (
            <TableRow key={item.id}>
              ${entity.fields.slice(0, 4).map(f => `<TableCell>{item.${f.name}}</TableCell>`).join('\n              ')}
              <TableCell>
                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => navigate(\`/${blueprint.slug}/${entity.name.toLowerCase()}/\${item.id}\`)}
                  >
                    <Edit className="h-4 w-4" />
                  </Button>
                  <Button
                    variant="destructive"
                    size="sm"
                    onClick={() => handleDelete(item.id)}
                  >
                    <Trash2 className="h-4 w-4" />
                  </Button>
                </div>
              </TableCell>
            </TableRow>
          ))}
        </TableBody>
      </Table>
    </div>
  );
};

export default ${entity.name}List;
`;
  }

  private generateReactFormPage(entity: BlueprintEntity, blueprint: ModuleBlueprint): string {
    const hookName = `use${entity.name}`;

    return `import { useEffect } from 'react';
import { useNavigate, useParams } from 'react-router-dom';
import { useForm } from 'react-hook-form';
import { ${hookName} } from '@/hooks/${hookName}';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';

const ${entity.name}Form = () => {
  const navigate = useNavigate();
  const { id } = useParams();
  const { ${entity.name.toLowerCase()}, create${entity.name}, update${entity.name} } = ${hookName}(id);
  
  const { register, handleSubmit, reset, formState: { errors } } = useForm();

  useEffect(() => {
    if (${entity.name.toLowerCase()}) {
      reset(${entity.name.toLowerCase()});
    }
  }, [${entity.name.toLowerCase()}, reset]);

  const onSubmit = async (data: any) => {
    try {
      if (id) {
        await update${entity.name}.mutateAsync({ id, ...data });
      } else {
        await create${entity.name}.mutateAsync(data);
      }
      navigate('/${blueprint.slug}/${entity.name.toLowerCase()}');
    } catch (error) {
      console.error('Error saving ${entity.name.toLowerCase()}:', error);
    }
  };

  return (
    <div className="container mx-auto p-6 max-w-2xl">
      <Card>
        <CardHeader>
          <CardTitle>{id ? 'Edit' : 'Create'} ${entity.displayName}</CardTitle>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit(onSubmit)} className="space-y-4">
            ${entity.fields
              .filter(f => f.name !== 'id')
              .map(field => `
            <div>
              <Label htmlFor="${field.name}">${field.name}</Label>
              <Input
                id="${field.name}"
                type="${field.type === 'number' ? 'number' : field.type === 'date' ? 'date' : 'text'}"
                {...register('${field.name}', { required: ${field.required} })}
              />
              {errors.${field.name} && <span className="text-sm text-destructive">This field is required</span>}
            </div>`)
              .join('\n')}

            <div className="flex gap-4">
              <Button type="submit">Save</Button>
              <Button type="button" variant="outline" onClick={() => navigate('/${blueprint.slug}/${entity.name.toLowerCase()}')}>
                Cancel
              </Button>
            </div>
          </form>
        </CardContent>
      </Card>
    </div>
  );
};

export default ${entity.name}Form;
`;
  }

  private generateReactViewPage(entity: BlueprintEntity, blueprint: ModuleBlueprint): string {
    const hookName = `use${entity.name}`;

    return `import { useNavigate, useParams } from 'react-router-dom';
import { ${hookName} } from '@/hooks/${hookName}';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { ArrowLeft, Edit } from 'lucide-react';

const ${entity.name}View = () => {
  const navigate = useNavigate();
  const { id } = useParams();
  const { ${entity.name.toLowerCase()}, isLoading } = ${hookName}(id);

  if (isLoading) {
    return <div>Loading...</div>;
  }

  if (!${entity.name.toLowerCase()}) {
    return <div>Not found</div>;
  }

  return (
    <div className="container mx-auto p-6 max-w-2xl">
      <div className="flex justify-between items-center mb-6">
        <Button variant="outline" onClick={() => navigate('/${blueprint.slug}/${entity.name.toLowerCase()}')}>
          <ArrowLeft className="h-4 w-4 mr-2" />
          Back
        </Button>
        <Button onClick={() => navigate(\`/${blueprint.slug}/${entity.name.toLowerCase()}/\${id}/edit\`)}>
          <Edit className="h-4 w-4 mr-2" />
          Edit
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>${entity.displayName} Details</CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          ${entity.fields.map(field => `
          <div>
            <div className="text-sm font-medium text-muted-foreground">${field.name}</div>
            <div className="text-base">{${entity.name.toLowerCase()}.${field.name}}</div>
          </div>`).join('\n')}
        </CardContent>
      </Card>
    </div>
  );
};

export default ${entity.name}View;
`;
  }

  private generateReactHook(entity: BlueprintEntity, blueprint: ModuleBlueprint): string {
    const pluralName = entity.name.toLowerCase() + 's';

    return `import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import { supabase } from '@/integrations/supabase/client';
import { useToast } from '@/hooks/use-toast';

export const use${entity.name} = (id?: string) => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  // Fetch all
  const { data: ${pluralName}, isLoading } = useQuery({
    queryKey: ['${pluralName}'],
    queryFn: async () => {
      const { data, error } = await supabase
        .from('${pluralName}')
        .select('*')
        .order('created_at', { ascending: false });

      if (error) throw error;
      return data;
    },
    enabled: !id,
  });

  // Fetch single
  const { data: ${entity.name.toLowerCase()} } = useQuery({
    queryKey: ['${entity.name.toLowerCase()}', id],
    queryFn: async () => {
      if (!id) return null;
      const { data, error } = await supabase
        .from('${pluralName}')
        .select('*')
        .eq('id', id)
        .single();

      if (error) throw error;
      return data;
    },
    enabled: !!id,
  });

  // Create
  const create${entity.name} = useMutation({
    mutationFn: async (values: any) => {
      const { data, error } = await supabase
        .from('${pluralName}')
        .insert(values)
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['${pluralName}'] });
      toast({ title: '${entity.displayName} created successfully' });
    },
    onError: (error: Error) => {
      toast({ title: 'Error', description: error.message, variant: 'destructive' });
    },
  });

  // Update
  const update${entity.name} = useMutation({
    mutationFn: async ({ id, ...values }: any) => {
      const { data, error } = await supabase
        .from('${pluralName}')
        .update(values)
        .eq('id', id)
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['${pluralName}'] });
      toast({ title: '${entity.displayName} updated successfully' });
    },
    onError: (error: Error) => {
      toast({ title: 'Error', description: error.message, variant: 'destructive' });
    },
  });

  // Delete
  const delete${entity.name} = useMutation({
    mutationFn: async (id: string) => {
      const { error } = await supabase
        .from('${pluralName}')
        .delete()
        .eq('id', id);

      if (error) throw error;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ['${pluralName}'] });
      toast({ title: '${entity.displayName} deleted successfully' });
    },
    onError: (error: Error) => {
      toast({ title: 'Error', description: error.message, variant: 'destructive' });
    },
  });

  return {
    ${pluralName},
    ${entity.name.toLowerCase()},
    isLoading,
    create${entity.name},
    update${entity.name},
    delete${entity.name},
  };
};
`;
  }

  private generateTypeScript(blueprint: ModuleBlueprint): string {
    const types = blueprint.entities.map(entity => {
      const fields = entity.fields.map(field => {
        const type = field.type === 'number' ? 'number' : 
                    field.type === 'boolean' ? 'boolean' : 
                    field.type === 'date' ? 'Date' :
                    field.type === 'json' ? 'any' : 'string';
        return `  ${field.name}${field.required ? '' : '?'}: ${type};`;
      }).join('\n');

      return `export interface ${entity.name} {
${fields}
  created_at?: string;
  updated_at?: string;
}`;
    }).join('\n\n');

    return `// Generated types for ${blueprint.name} module\n\n${types}\n`;
  }

  private generateNext(blueprint: ModuleBlueprint): Record<string, string> {
    return {
      'README.md': '# Next.js frontend generation coming soon',
    };
  }

  private generateVue(blueprint: ModuleBlueprint): Record<string, string> {
    return {
      'README.md': '# Vue frontend generation coming soon',
    };
  }
}

export const frontendGenerator = new FrontendGenerator();
