// Phase 3: Natural Language to Blueprint Converter

import { ModuleBlueprint, BlueprintEntity, BlueprintField } from '@/types/moduleBlueprint';

export class NaturalToBlueprint {
  /**
   * Parse natural language description into a structured blueprint
   */
  async parseDescription(description: string, locale: string = 'en'): Promise<ModuleBlueprint> {
    console.log('🧠 Parsing natural description to blueprint...', description);

    // Extract module name and slug
    const name = this.extractModuleName(description);
    const slug = this.generateSlug(name);

    // Extract entities
    const entities = this.extractEntities(description, locale);

    // Extract relationships
    const relationships = this.extractRelationships(description, entities);

    // Generate UI pages
    const uiPages = this.generateUIPages(entities);

    // Generate API routes
    const apiRoutes = this.generateAPIRoutes(entities);

    // Detect complexity
    const complexity = this.detectComplexity(entities, relationships);

    // Detect tech stack preferences
    const { backendStack, frontendStack } = this.detectTechStack(description);

    const blueprint: ModuleBlueprint = {
      name,
      slug,
      description,
      naturalDescription: description,
      entities,
      relationships,
      uiPages,
      apiRoutes,
      backendStack,
      frontendStack,
      permissions: this.generatePermissions(entities),
      menuItems: this.generateMenuItems(name, slug, entities),
      complexity,
      status: 'draft',
      version: '1.0.0',
    };

    console.log('✅ Blueprint created:', blueprint);
    return blueprint;
  }

  private extractModuleName(description: string): string {
    // Extract from patterns like "Build a X system" or "Create X module"
    const patterns = [
      /(?:build|create|make)\s+(?:a\s+)?([a-z\s]+?)(?:\s+system|\s+module|\s+app)/i,
      /^([A-Z][a-z]+(?:\s+[A-Z][a-z]+)*)/,
    ];

    for (const pattern of patterns) {
      const match = description.match(pattern);
      if (match) {
        return this.capitalize(match[1].trim());
      }
    }

    return 'Custom Module';
  }

  private generateSlug(name: string): string {
    return name.toLowerCase().replace(/\s+/g, '-').replace(/[^a-z0-9-]/g, '');
  }

  private extractEntities(description: string, locale: string): BlueprintEntity[] {
    const entities: BlueprintEntity[] = [];

    // Common entity keywords
    const entityKeywords = ['user', 'product', 'order', 'customer', 'post', 'article', 'category', 'tag'];
    
    // Bangla keywords
    const banglaKeywords = ['ব্যবহারকারী', 'পণ্য', 'অর্ডার', 'গ্রাহক', 'পোস্ট'];

    const keywords = locale === 'bn' ? [...entityKeywords, ...banglaKeywords] : entityKeywords;

    keywords.forEach(keyword => {
      if (description.toLowerCase().includes(keyword)) {
        const entity: BlueprintEntity = {
          name: this.capitalize(keyword),
          displayName: this.capitalize(keyword),
          fields: this.generateDefaultFields(keyword),
          hasTimestamps: true,
          hasSoftDelete: false,
          hasOwnership: true,
        };
        entities.push(entity);
      }
    });

    // If no entities found, create a default one
    if (entities.length === 0) {
      entities.push({
        name: 'Item',
        displayName: 'Item',
        fields: this.generateDefaultFields('item'),
        hasTimestamps: true,
        hasSoftDelete: false,
        hasOwnership: true,
      });
    }

    return entities;
  }

  private generateDefaultFields(entityName: string): BlueprintField[] {
    const baseFields: BlueprintField[] = [
      { name: 'id', type: 'string', required: true, isUnique: true },
      { name: 'title', type: 'string', required: true, isSearchable: true },
      { name: 'description', type: 'text', required: false },
    ];

    // Entity-specific fields
    const specificFields: Record<string, BlueprintField[]> = {
      user: [
        { name: 'email', type: 'string', required: true, isUnique: true },
        { name: 'name', type: 'string', required: true },
        { name: 'role', type: 'string', required: false, defaultValue: 'user' },
      ],
      product: [
        { name: 'price', type: 'number', required: true },
        { name: 'stock', type: 'number', required: false, defaultValue: 0 },
        { name: 'category', type: 'reference', required: false, referenceEntity: 'Category' },
      ],
      order: [
        { name: 'total', type: 'number', required: true },
        { name: 'status', type: 'string', required: true, defaultValue: 'pending' },
        { name: 'customer', type: 'reference', required: true, referenceEntity: 'User' },
      ],
    };

    return specificFields[entityName.toLowerCase()] || baseFields;
  }

  private extractRelationships(description: string, entities: BlueprintEntity[]) {
    const relationships = [];

    // Detect common relationship patterns
    if (entities.some(e => e.name === 'Product') && entities.some(e => e.name === 'Category')) {
      relationships.push({
        type: 'one-to-many' as const,
        from: 'Category',
        to: 'Product',
      });
    }

    if (entities.some(e => e.name === 'Order') && entities.some(e => e.name === 'User')) {
      relationships.push({
        type: 'one-to-many' as const,
        from: 'User',
        to: 'Order',
      });
    }

    return relationships;
  }

  private generateUIPages(entities: BlueprintEntity[]) {
    const pages = [];

    entities.forEach(entity => {
      // List page
      pages.push({
        type: 'list' as const,
        entity: entity.name,
        title: `${entity.displayName} List`,
        route: `/${this.generateSlug(entity.name)}`,
        permissions: ['read'],
        fields: entity.fields.filter(f => f.isSearchable).map(f => f.name),
      });

      // Form page
      pages.push({
        type: 'form' as const,
        entity: entity.name,
        title: `${entity.displayName} Form`,
        route: `/${this.generateSlug(entity.name)}/edit`,
        permissions: ['create', 'update'],
      });

      // View page
      pages.push({
        type: 'view' as const,
        entity: entity.name,
        title: `${entity.displayName} Details`,
        route: `/${this.generateSlug(entity.name)}/:id`,
        permissions: ['read'],
      });
    });

    return pages;
  }

  private generateAPIRoutes(entities: BlueprintEntity[]) {
    const routes = [];

    entities.forEach(entity => {
      const basePath = `/${this.generateSlug(entity.name)}`;

      routes.push(
        { method: 'GET' as const, path: basePath, entity: entity.name, action: 'list', auth: true, permissions: ['read'] },
        { method: 'GET' as const, path: `${basePath}/:id`, entity: entity.name, action: 'show', auth: true, permissions: ['read'] },
        { method: 'POST' as const, path: basePath, entity: entity.name, action: 'create', auth: true, permissions: ['create'] },
        { method: 'PUT' as const, path: `${basePath}/:id`, entity: entity.name, action: 'update', auth: true, permissions: ['update'] },
        { method: 'DELETE' as const, path: `${basePath}/:id`, entity: entity.name, action: 'delete', auth: true, permissions: ['delete'] }
      );
    });

    return routes;
  }

  private generatePermissions(entities: BlueprintEntity[]): string[] {
    const permissions = new Set<string>();
    
    entities.forEach(entity => {
      permissions.add(`${entity.name.toLowerCase()}.read`);
      permissions.add(`${entity.name.toLowerCase()}.create`);
      permissions.add(`${entity.name.toLowerCase()}.update`);
      permissions.add(`${entity.name.toLowerCase()}.delete`);
    });

    return Array.from(permissions);
  }

  private generateMenuItems(name: string, slug: string, entities: BlueprintEntity[]) {
    return [
      {
        label: name,
        route: `/${slug}`,
        icon: 'Package',
        order: 100,
      },
      ...entities.map((entity, index) => ({
        label: entity.displayName,
        route: `/${slug}/${this.generateSlug(entity.name)}`,
        icon: 'List',
        order: 100 + index + 1,
        parent: name,
      })),
    ];
  }

  private detectComplexity(entities: BlueprintEntity[], relationships: any[]): 'simple' | 'medium' | 'complex' {
    const entityCount = entities.length;
    const relationshipCount = relationships.length;
    const totalFields = entities.reduce((sum, e) => sum + e.fields.length, 0);

    if (entityCount <= 2 && relationshipCount === 0 && totalFields <= 10) {
      return 'simple';
    } else if (entityCount <= 5 && relationshipCount <= 3 && totalFields <= 30) {
      return 'medium';
    } else {
      return 'complex';
    }
  }

  private detectTechStack(description: string): { backendStack: 'laravel' | 'dotnet' | 'express'; frontendStack: 'react' | 'next' | 'vue' } {
    const lower = description.toLowerCase();

    let backendStack: 'laravel' | 'dotnet' | 'express' = 'laravel';
    if (lower.includes('.net') || lower.includes('dotnet')) backendStack = 'dotnet';
    if (lower.includes('express') || lower.includes('node')) backendStack = 'express';

    let frontendStack: 'react' | 'next' | 'vue' = 'react';
    if (lower.includes('next') || lower.includes('nextjs')) frontendStack = 'next';
    if (lower.includes('vue')) frontendStack = 'vue';

    return { backendStack, frontendStack };
  }

  private capitalize(str: string): string {
    return str.charAt(0).toUpperCase() + str.slice(1);
  }
}

export const naturalToBlueprint = new NaturalToBlueprint();
