export interface ModuleConfig {
  name: string;
  slug: string;
  icon: string;
  enabled: boolean;
  description?: string;
  version?: string;
  author?: string;
}

class ModuleLoader {
  private modules: Map<string, ModuleConfig> = new Map();
  private registeredModules: ModuleConfig[] = [];

  constructor() {
    this.loadModules();
  }

  private loadModules() {
    // Load from localStorage
    const stored = localStorage.getItem("registered_modules");
    if (stored) {
      this.registeredModules = JSON.parse(stored);
      this.registeredModules.forEach((mod) => {
        this.modules.set(mod.slug, mod);
      });
    }

    // Register default modules
    this.registerDefaultModules();
  }

  private registerDefaultModules() {
    const defaults: ModuleConfig[] = [
      {
        name: "Smart Academy",
        slug: "smart-academy",
        icon: "GraduationCap",
        enabled: true,
        description: "Education and learning platform",
      },
      {
        name: "Smart Bazar",
        slug: "smart-bazar",
        icon: "ShoppingBag",
        enabled: true,
        description: "E-commerce marketplace",
      },
      {
        name: "Smart Doctor",
        slug: "smart-doctor",
        icon: "Stethoscope",
        enabled: true,
        description: "Healthcare management",
      },
      {
        name: "Smart Job",
        slug: "smart-job",
        icon: "Briefcase",
        enabled: true,
        description: "Job portal and recruitment",
      },
      {
        name: "Smart Pay",
        slug: "smart-pay",
        icon: "Wallet",
        enabled: true,
        description: "Payment and wallet system",
      },
    ];

    defaults.forEach((mod) => {
      if (!this.modules.has(mod.slug)) {
        this.registerModule(mod);
      }
    });
  }

  registerModule(config: ModuleConfig): void {
    this.modules.set(config.slug, config);
    this.registeredModules = Array.from(this.modules.values());
    this.saveModules();
  }

  loadModule(slug: string): ModuleConfig | undefined {
    return this.modules.get(slug);
  }

  listModules(): ModuleConfig[] {
    return this.registeredModules;
  }

  enableModule(slug: string): void {
    const module = this.modules.get(slug);
    if (module) {
      module.enabled = true;
      this.saveModules();
    }
  }

  disableModule(slug: string): void {
    const module = this.modules.get(slug);
    if (module) {
      module.enabled = false;
      this.saveModules();
    }
  }

  generateModule(templateName: string): ModuleConfig {
    const timestamp = Date.now();
    const newModule: ModuleConfig = {
      name: `Generated ${templateName}`,
      slug: `generated-${templateName.toLowerCase()}-${timestamp}`,
      icon: "Package",
      enabled: true,
      description: `Auto-generated module based on ${templateName} template`,
      version: "1.0.0",
      author: "Smart Friend AI",
    };

    this.registerModule(newModule);
    return newModule;
  }

  private saveModules() {
    localStorage.setItem("registered_modules", JSON.stringify(this.registeredModules));
  }
}

export const moduleLoader = new ModuleLoader();
