export type VoiceProfile = "male" | "female" | "neutral";

export class VoiceOutput {
  private isSpeaking = false;
  private currentUtterance: SpeechSynthesisUtterance | null = null;

  isSupported(): boolean {
    return "speechSynthesis" in window;
  }

  speak(text: string, profile: VoiceProfile = "neutral"): void {
    if (!this.isSupported()) {
      console.warn("Speech synthesis is not supported");
      return;
    }

    // Stop any ongoing speech
    this.stop();

    const utterance = new SpeechSynthesisUtterance(text);
    this.currentUtterance = utterance;

    // Configure voice based on profile
    const voices = speechSynthesis.getVoices();
    let selectedVoice = null;

    if (profile === "male") {
      selectedVoice = voices.find((v) => v.name.toLowerCase().includes("male")) || voices[0];
    } else if (profile === "female") {
      selectedVoice = voices.find((v) => v.name.toLowerCase().includes("female")) || voices[1];
    } else {
      selectedVoice = voices[0];
    }

    if (selectedVoice) {
      utterance.voice = selectedVoice;
    }

    utterance.rate = 0.9;
    utterance.pitch = 1.0;
    utterance.volume = 1.0;

    utterance.onstart = () => {
      this.isSpeaking = true;
    };

    utterance.onend = () => {
      this.isSpeaking = false;
      this.currentUtterance = null;
    };

    utterance.onerror = () => {
      this.isSpeaking = false;
      this.currentUtterance = null;
    };

    speechSynthesis.speak(utterance);
  }

  stop(): void {
    if (this.isSupported() && this.isSpeaking) {
      speechSynthesis.cancel();
      this.isSpeaking = false;
      this.currentUtterance = null;
    }
  }

  getIsSpeaking(): boolean {
    return this.isSpeaking;
  }
}

export const voiceOutput = new VoiceOutput();
