import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export interface AIBrainRequest {
  prompt: string;
  context?: string;
  filePath?: string;
  action: "generate" | "refactor" | "fix" | "optimize" | "explain";
  language?: string;
}

export interface AIBrainResponse {
  code: string;
  explanation: string;
  suggestions?: string[];
}

export const useAIBrain = (projectId: string) => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const generateCode = useMutation({
    mutationFn: async (request: AIBrainRequest) => {
      const { data, error } = await supabase.functions.invoke("ai-code-generator", {
        body: {
          prompt: request.prompt,
          context: request.context,
          action: request.action,
          language: request.language || "typescript",
        },
      });

      if (error) throw error;

      // Save to ai_changes table
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      if (request.filePath) {
        await supabase.from("ai_changes").insert({
          user_id: user.id,
          project_id: projectId,
          file_path: request.filePath,
          original_code: request.context,
          generated_code: data.generatedCode,
          prompt: request.prompt,
          applied: false,
        });
      }

      return {
        code: data.generatedCode,
        explanation: data.explanation || "Code generated successfully",
        suggestions: data.suggestions || [],
      } as AIBrainResponse;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["ai_changes", projectId] });
      toast({ title: "AI Brain completed successfully" });
    },
    onError: (error: Error) => {
      toast({ 
        title: "AI Brain Error", 
        description: error.message, 
        variant: "destructive" 
      });
    },
  });

  const { data: history = [] } = useQuery({
    queryKey: ["ai_changes", projectId],
    queryFn: async () => {
      const { data, error } = await supabase
        .from("ai_changes")
        .select("*")
        .eq("project_id", projectId)
        .order("created_at", { ascending: false })
        .limit(50);

      if (error) throw error;
      return data;
    },
  });

  return {
    generateCode,
    history,
    isGenerating: generateCode.isPending,
  };
};
