import { useMutation, useQuery, useQueryClient } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export interface AIChange {
  id: string;
  user_id: string;
  project_id: string;
  file_path: string;
  original_code?: string;
  generated_code: string;
  prompt: string;
  applied: boolean;
  created_at: string;
}

export const useAIBuilder = (projectId: string) => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: changes = [] } = useQuery({
    queryKey: ["ai_changes", projectId],
    queryFn: async () => {
      const { data, error } = await supabase
        .from("ai_changes")
        .select("*")
        .eq("project_id", projectId)
        .order("created_at", { ascending: false });

      if (error) throw error;
      return data as AIChange[];
    },
  });

  const generateCode = useMutation({
    mutationFn: async ({ prompt, filePath, originalCode }: {
      prompt: string;
      filePath: string;
      originalCode?: string;
    }) => {
      const { data, error } = await supabase.functions.invoke("ai-code-generator", {
        body: { prompt, context: originalCode },
      });

      if (error) throw error;

      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data: change, error: insertError } = await supabase
        .from("ai_changes")
        .insert({
          user_id: user.id,
          project_id: projectId,
          file_path: filePath,
          original_code: originalCode,
          generated_code: data.generatedCode,
          prompt,
          applied: false,
        })
        .select()
        .single();

      if (insertError) throw insertError;
      return { change, generatedCode: data.generatedCode };
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["ai_changes", projectId] });
      toast({ title: "Code generated successfully" });
    },
    onError: (error: Error) => {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    },
  });

  const applyChange = useMutation({
    mutationFn: async (changeId: string) => {
      const { data, error } = await supabase
        .from("ai_changes")
        .update({ applied: true })
        .eq("id", changeId)
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["ai_changes", projectId] });
      toast({ title: "Change applied successfully" });
    },
    onError: (error: Error) => {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    },
  });

  return {
    changes,
    generateCode,
    applyChange,
  };
};
