import { useState, useRef, useCallback } from "react";
import { supabase } from "@/integrations/supabase/client";
import { toast } from "sonner";

export interface CallBotConfig {
  language: string;
  voice: string;
  systemPrompt: string;
  enableSentiment: boolean;
  enableTicketing: boolean;
}

export interface CallIntent {
  type: "support" | "inquiry" | "complaint" | "feedback" | "other";
  confidence: number;
  summary: string;
}

export interface CallSentiment {
  emotion: "positive" | "neutral" | "negative" | "frustrated" | "satisfied";
  score: number;
}

export interface CallTranscript {
  timestamp: string;
  speaker: "user" | "bot";
  text: string;
}

export const useAICallBot = () => {
  const [active, setActive] = useState(false);
  const [connecting, setConnecting] = useState(false);
  const [speaking, setSpeaking] = useState(false);
  const [transcript, setTranscript] = useState<CallTranscript[]>([]);
  const [intent, setIntent] = useState<CallIntent | null>(null);
  const [sentiment, setSentiment] = useState<CallSentiment | null>(null);
  
  const wsRef = useRef<WebSocket | null>(null);
  const audioContextRef = useRef<AudioContext | null>(null);

  const startBot = useCallback(async (config: CallBotConfig) => {
    setConnecting(true);
    toast.info("Initializing AI Call Bot...");

    try {
      // Get ephemeral token from edge function
      const { data, error } = await supabase.functions.invoke("ai-call-bot-token", {
        body: { 
          language: config.language,
          voice: config.voice,
          systemPrompt: config.systemPrompt
        }
      });

      if (error) throw error;

      // Initialize audio context
      audioContextRef.current = new AudioContext({ sampleRate: 24000 });

      // Connect to WebSocket relay
      const wsUrl = `${import.meta.env.VITE_SUPABASE_URL}/functions/v1/ai-call-bot`.replace('https://', 'wss://');
      wsRef.current = new WebSocket(wsUrl);

      wsRef.current.onopen = () => {
        console.log("AI Call Bot connected");
        setActive(true);
        setConnecting(false);
        toast.success("AI Call Bot ready");
        
        // Send initial config
        wsRef.current?.send(JSON.stringify({
          type: "config",
          token: data.client_secret.value
        }));
      };

      wsRef.current.onmessage = async (event) => {
        const message = JSON.parse(event.data);
        
        switch (message.type) {
          case "audio.delta":
            setSpeaking(true);
            await playAudioChunk(message.audio);
            break;
            
          case "audio.done":
            setSpeaking(false);
            break;
            
          case "transcript":
            addTranscript(message.speaker, message.text);
            break;
            
          case "intent":
            setIntent(message.intent);
            break;
            
          case "sentiment":
            setSentiment(message.sentiment);
            break;
            
          case "ticket.created":
            toast.success("Support ticket created automatically");
            break;
        }
      };

      wsRef.current.onerror = (error) => {
        console.error("WebSocket error:", error);
        toast.error("Connection error");
      };

      wsRef.current.onclose = () => {
        setActive(false);
        setSpeaking(false);
        toast.info("AI Call Bot disconnected");
      };

    } catch (error: any) {
      console.error("Failed to start bot:", error);
      toast.error(`Failed to start: ${error.message}`);
      setConnecting(false);
    }
  }, []);

  const stopBot = useCallback(() => {
    wsRef.current?.close();
    audioContextRef.current?.close();
    setActive(false);
    setSpeaking(false);
    toast.info("AI Call Bot stopped");
  }, []);

  const sendAudio = useCallback((audioData: Float32Array) => {
    if (!wsRef.current || wsRef.current.readyState !== WebSocket.OPEN) return;

    // Convert to base64
    const int16Array = new Int16Array(audioData.length);
    for (let i = 0; i < audioData.length; i++) {
      const s = Math.max(-1, Math.min(1, audioData[i]));
      int16Array[i] = s < 0 ? s * 0x8000 : s * 0x7FFF;
    }

    const uint8Array = new Uint8Array(int16Array.buffer);
    const base64 = btoa(String.fromCharCode(...uint8Array));

    wsRef.current.send(JSON.stringify({
      type: "audio.input",
      audio: base64
    }));
  }, []);

  const playAudioChunk = async (base64Audio: string) => {
    if (!audioContextRef.current) return;

    try {
      // Decode base64 to bytes
      const binaryString = atob(base64Audio);
      const bytes = new Uint8Array(binaryString.length);
      for (let i = 0; i < binaryString.length; i++) {
        bytes[i] = binaryString.charCodeAt(i);
      }

      // Convert to audio buffer
      const audioBuffer = await audioContextRef.current.decodeAudioData(bytes.buffer);
      
      // Play
      const source = audioContextRef.current.createBufferSource();
      source.buffer = audioBuffer;
      source.connect(audioContextRef.current.destination);
      source.start(0);
    } catch (error) {
      console.error("Audio playback error:", error);
    }
  };

  const addTranscript = (speaker: "user" | "bot", text: string) => {
    setTranscript(prev => [...prev, {
      timestamp: new Date().toISOString(),
      speaker,
      text
    }]);
  };

  const saveCallRecord = useCallback(async (userId: string, duration: number) => {
    try {
      const { error } = await supabase.from("call_history").insert({
        user_id: userId,
        call_duration: duration,
        transcript: transcript.map(t => `${t.speaker}: ${t.text}`).join("\n"),
        ai_summary: intent?.summary || "Call completed"
      });

      if (error) throw error;
      toast.success("Call record saved");
    } catch (error: any) {
      console.error("Failed to save call record:", error);
      toast.error("Failed to save call record");
    }
  }, [transcript, intent]);

  return {
    startBot,
    stopBot,
    sendAudio,
    saveCallRecord,
    active,
    connecting,
    speaking,
    transcript,
    intent,
    sentiment
  };
};
