import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export interface AffiliateLink {
  id: string;
  user_id: string;
  referral_code: string;
  module_slug?: string;
  clicks: number;
  conversions: number;
  total_earnings: number;
  created_at: string;
  updated_at: string;
}

export interface AffiliateConversion {
  id: string;
  affiliate_link_id: string;
  referred_user_id?: string;
  module_slug?: string;
  commission_amount: number;
  status: string;
  created_at: string;
}

export interface AffiliatePayout {
  id: string;
  user_id: string;
  amount: number;
  status: string;
  payment_method?: string;
  payment_details?: any;
  processed_at?: string;
  created_at: string;
}

export const useAffiliate = () => {
  const { toast } = useToast();
  const queryClient = useQueryClient();

  const { data: affiliateLink } = useQuery({
    queryKey: ["affiliate_link"],
    queryFn: async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data, error } = await supabase
        .from("affiliate_links")
        .select("*")
        .eq("user_id", user.id)
        .maybeSingle();

      if (error && error.code !== "PGRST116") throw error;
      return data as AffiliateLink | null;
    },
  });

  const { data: conversions = [] } = useQuery({
    queryKey: ["affiliate_conversions"],
    queryFn: async () => {
      if (!affiliateLink) return [];

      const { data, error } = await supabase
        .from("affiliate_conversions")
        .select("*")
        .eq("affiliate_link_id", affiliateLink.id)
        .order("created_at", { ascending: false });

      if (error) throw error;
      return data as AffiliateConversion[];
    },
    enabled: !!affiliateLink,
  });

  const { data: payouts = [] } = useQuery({
    queryKey: ["affiliate_payouts"],
    queryFn: async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data, error } = await supabase
        .from("affiliate_payouts")
        .select("*")
        .eq("user_id", user.id)
        .order("created_at", { ascending: false });

      if (error) throw error;
      return data as AffiliatePayout[];
    },
  });

  const generateLink = useMutation({
    mutationFn: async (moduleSlug?: string) => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const referralCode = `${user.id.slice(0, 8)}-${Date.now()}`;

      const { data, error } = await supabase
        .from("affiliate_links")
        .insert({
          user_id: user.id,
          referral_code: referralCode,
          module_slug: moduleSlug,
        })
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["affiliate_link"] });
      toast({ title: "Affiliate link generated successfully" });
    },
    onError: (error: Error) => {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    },
  });

  const requestPayout = useMutation({
    mutationFn: async ({ amount, paymentMethod }: { amount: number; paymentMethod: string }) => {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data, error } = await supabase
        .from("affiliate_payouts")
        .insert({
          user_id: user.id,
          amount,
          payment_method: paymentMethod,
          status: "pending",
        })
        .select()
        .single();

      if (error) throw error;
      return data;
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["affiliate_payouts"] });
      toast({ title: "Payout request submitted successfully" });
    },
    onError: (error: Error) => {
      toast({ title: "Error", description: error.message, variant: "destructive" });
    },
  });

  return {
    affiliateLink,
    conversions,
    payouts,
    generateLink,
    requestPayout,
  };
};