import { useState } from "react";
import { useToast } from "@/hooks/use-toast";

export type AgentStatus = "idle" | "busy" | "error" | "disabled";

export interface Agent {
  id: string;
  name: string;
  level: number;
  description: string;
  status: AgentStatus;
  currentTask?: string;
  lastActive?: Date;
  capabilities: string[];
}

export const useAgentManager = () => {
  const { toast } = useToast();

  const [agents] = useState<Agent[]>([
    { id: "sf1", name: "SmartFriend1", level: 1, description: "Owner's Boss Assistant - Command Router", status: "idle", capabilities: ["command_routing", "task_delegation", "system_overview"] },
    { id: "sf2", name: "SmartFriend2", level: 2, description: "Ava's Child-Safe Assistant", status: "idle", capabilities: ["child_safety", "education", "storytelling"] },
    
    // Level 3-25 Worker Agents
    { id: "web_dev", name: "WebDeveloperAgent", level: 3, description: "Frontend Development Specialist", status: "idle", capabilities: ["react", "typescript", "ui_components"] },
    { id: "laravel", name: "LaravelExpertAgent", level: 4, description: "Backend Development Expert", status: "idle", capabilities: ["laravel", "php", "api_development"] },
    { id: "bug_fix", name: "BugFixAgent", level: 5, description: "Error Detection & Repair", status: "idle", capabilities: ["debugging", "testing", "error_analysis"] },
    { id: "devops", name: "AutoDevOpsAgent", level: 6, description: "CI/CD & Deployment Automation", status: "idle", capabilities: ["deployment", "hosting", "monitoring"] },
    { id: "content", name: "ContentWriterAgent", level: 7, description: "Content Creation Specialist", status: "idle", capabilities: ["writing", "copywriting", "content_strategy"] },
    { id: "marketer", name: "SmartMarketerAgent", level: 8, description: "Marketing & Growth Expert", status: "idle", capabilities: ["marketing", "campaigns", "analytics"] },
    { id: "news", name: "NewsWriterAgent", level: 9, description: "News & Article Writer", status: "idle", capabilities: ["journalism", "news_writing", "research"] },
    { id: "seo", name: "SEOAssistantAgent", level: 10, description: "SEO Optimization Expert", status: "idle", capabilities: ["seo", "keywords", "optimization"] },
    { id: "designer", name: "DesignerAgent", level: 11, description: "Graphic Design Specialist", status: "idle", capabilities: ["design", "graphics", "branding"] },
    { id: "logo", name: "LogoCreatorAgent", level: 12, description: "Logo Design Expert", status: "idle", capabilities: ["logo_design", "branding", "identity"] },
    { id: "pdf", name: "PDFMakerAgent", level: 13, description: "Document & PDF Generator", status: "idle", capabilities: ["pdf_generation", "documents", "reports"] },
    { id: "photo", name: "PhotoEditorAgent", level: 14, description: "Image Editing Specialist", status: "idle", capabilities: ["photo_editing", "image_enhancement", "filters"] },
    { id: "video", name: "VideoEditorAgent", level: 15, description: "Video Production Expert", status: "idle", capabilities: ["video_editing", "motion_graphics", "production"] },
    { id: "script", name: "ScriptWriterAgent", level: 16, description: "Script & Screenplay Writer", status: "idle", capabilities: ["scriptwriting", "storytelling", "narrative"] },
    { id: "accountant", name: "AccountantAgent", level: 17, description: "Financial & Accounting Expert", status: "idle", capabilities: ["accounting", "finance", "reporting"] },
    { id: "lawyer", name: "LawyerDraftAgent", level: 18, description: "Legal Document Specialist", status: "idle", capabilities: ["legal_documents", "contracts", "agreements"] },
    { id: "doctor", name: "DoctorReportAgent", level: 19, description: "Medical Report Analyst", status: "idle", capabilities: ["medical_analysis", "health_reports", "diagnosis_review"] },
    { id: "tourist", name: "TouristInfoAgent", level: 20, description: "Travel & Tourism Specialist", status: "idle", capabilities: ["travel_planning", "tourism", "destinations"] },
    { id: "hospital", name: "HospitalReceptionAgent", level: 21, description: "Healthcare Service Coordinator", status: "idle", capabilities: ["appointment_scheduling", "patient_care", "hospital_services"] },
    { id: "translator", name: "TranslatorAgent", level: 22, description: "Multi-language Translation Expert", status: "idle", capabilities: ["translation", "localization", "languages"] },
    { id: "data", name: "DataAnalystAgent", level: 23, description: "Data Analysis & Insights", status: "idle", capabilities: ["data_analysis", "statistics", "insights"] },
    { id: "qa", name: "QualityAssuranceAgent", level: 24, description: "Quality Testing & Validation", status: "idle", capabilities: ["testing", "quality_assurance", "validation"] },
    { id: "security", name: "SecurityAuditAgent", level: 25, description: "Security Analysis & Auditing", status: "idle", capabilities: ["security_audit", "vulnerability_scan", "compliance"] },
  ]);

  const [agentStates, setAgentStates] = useState<Record<string, AgentStatus>>(
    agents.reduce((acc, agent) => ({ ...acc, [agent.id]: "idle" }), {})
  );

  const [agentTasks, setAgentTasks] = useState<Record<string, string>>({});

  const getAgent = (nameOrId: string): Agent | undefined => {
    return agents.find(a => 
      a.id === nameOrId || 
      a.name.toLowerCase() === nameOrId.toLowerCase() ||
      a.name.toLowerCase().replace("agent", "").trim() === nameOrId.toLowerCase()
    );
  };

  const setAgentStatus = (agentId: string, status: AgentStatus, task?: string) => {
    setAgentStates(prev => ({ ...prev, [agentId]: status }));
    if (task) {
      setAgentTasks(prev => ({ ...prev, [agentId]: task }));
    }
  };

  const enableAgent = (agentId: string) => {
    setAgentStatus(agentId, "idle");
    const agent = agents.find(a => a.id === agentId);
    toast({
      title: "Agent enabled",
      description: `${agent?.name} is now active`,
    });
  };

  const disableAgent = (agentId: string) => {
    setAgentStatus(agentId, "disabled");
    const agent = agents.find(a => a.id === agentId);
    toast({
      title: "Agent disabled",
      description: `${agent?.name} has been disabled`,
      variant: "destructive",
    });
  };

  const stopAll = () => {
    agents.forEach(agent => {
      if (agentStates[agent.id] === "busy") {
        setAgentStatus(agent.id, "idle");
      }
    });
    setAgentTasks({});
    toast({
      title: "All agents stopped",
      description: "All running tasks have been cancelled",
      variant: "destructive",
    });
  };

  const getStatusSummary = () => {
    const idle = agents.filter(a => agentStates[a.id] === "idle").length;
    const busy = agents.filter(a => agentStates[a.id] === "busy").length;
    const error = agents.filter(a => agentStates[a.id] === "error").length;
    const disabled = agents.filter(a => agentStates[a.id] === "disabled").length;

    return { idle, busy, error, disabled, total: agents.length };
  };

  const getAgentsByLevel = () => {
    return {
      boss: agents.slice(0, 2), // SmartFriend 1 & 2
      workers: agents.slice(2),  // Level 3-25
    };
  };

  return {
    agents,
    agentStates,
    agentTasks,
    getAgent,
    setAgentStatus,
    enableAgent,
    disableAgent,
    stopAll,
    getStatusSummary,
    getAgentsByLevel,
  };
};
