import { useState, useCallback } from "react";
import { useToast } from "@/hooks/use-toast";
import { useErrorScanner, ErrorLocation } from "@/hooks/useErrorScanner";

export interface BugReport {
  id: string;
  errorMessage: string;
  affectedFiles: string[];
  severity: "critical" | "error" | "warning" | "info";
  stackTrace?: string;
  suggestedFixes: string[];
  groupId?: string;
  occurrences: number;
  firstSeen: Date;
  lastSeen: Date;
}

export const useAutoBugFinder = () => {
  const [scanning, setScanning] = useState(false);
  const [bugs, setBugs] = useState<BugReport[]>([]);
  const [fixing, setFixing] = useState<string | null>(null);
  const { toast } = useToast();
  const { scanProject, errors } = useErrorScanner();

  const scanForErrors = useCallback(async (projectPath?: string): Promise<BugReport[]> => {
    setScanning(true);
    
    try {
      toast({
        title: "Scanning for Errors",
        description: "Analyzing project for bugs and issues...",
      });

      const scanResults = await scanProject();

      // Convert error locations to bug reports
      const bugReports: BugReport[] = scanResults.map((error, index) => ({
        id: `bug_${Date.now()}_${index}`,
        errorMessage: error.message,
        affectedFiles: [error.file],
        severity: error.severity,
        stackTrace: error.line ? `Line ${error.line}${error.column ? `, Column ${error.column}` : ''}` : undefined,
        suggestedFixes: generateSuggestedFixes(error),
        occurrences: 1,
        firstSeen: new Date(),
        lastSeen: new Date(),
      }));

      // Group similar errors
      const grouped = groupSimilarErrors(bugReports);

      setBugs(grouped);

      toast({
        title: "Scan Complete",
        description: `Found ${grouped.length} unique issues`,
      });

      return grouped;
    } catch (error: any) {
      toast({
        title: "Scan Failed",
        description: error.message,
        variant: "destructive",
      });
      return [];
    } finally {
      setScanning(false);
    }
  }, [scanProject, toast]);

  const generateSuggestedFixes = (error: ErrorLocation): string[] => {
    const fixes: string[] = [];

    switch (error.type) {
      case "syntax":
        fixes.push("Fix syntax error by checking for missing brackets or semicolons");
        fixes.push("Verify code structure and formatting");
        break;
      case "import":
        fixes.push("Update import path to correct location");
        fixes.push("Check if imported module exists");
        fixes.push("Install missing dependency");
        break;
      case "undefined":
        fixes.push("Add null/undefined check before accessing property");
        fixes.push("Initialize variable with default value");
        fixes.push("Use optional chaining (?.)");
        break;
      case "runtime":
        fixes.push("Add error boundary to catch runtime errors");
        fixes.push("Add validation before processing data");
        fixes.push("Handle edge cases and null values");
        break;
      case "routing":
        fixes.push("Verify route path is correctly defined");
        fixes.push("Check route component exists");
        fixes.push("Update route configuration");
        break;
      case "component":
        fixes.push("Check component props and types");
        fixes.push("Verify component lifecycle methods");
        fixes.push("Review component dependencies");
        break;
    }

    if (error.severity === "critical") {
      fixes.unshift("CRITICAL: Fix immediately to prevent application failure");
    }

    return fixes;
  };

  const groupSimilarErrors = (bugReports: BugReport[]): BugReport[] => {
    const grouped = new Map<string, BugReport>();

    bugReports.forEach(bug => {
      // Create a key based on error message (first 50 chars) and severity
      const key = `${bug.errorMessage.substring(0, 50)}_${bug.severity}`;
      
      if (grouped.has(key)) {
        const existing = grouped.get(key)!;
        existing.occurrences += 1;
        existing.affectedFiles = [...new Set([...existing.affectedFiles, ...bug.affectedFiles])];
        existing.lastSeen = new Date();
      } else {
        grouped.set(key, { ...bug, groupId: key });
      }
    });

    return Array.from(grouped.values()).sort((a, b) => {
      const severityOrder = { critical: 0, error: 1, warning: 2, info: 3 };
      return severityOrder[a.severity] - severityOrder[b.severity];
    });
  };

  const applyAutoFix = useCallback(async (bugId: string): Promise<boolean> => {
    const bug = bugs.find(b => b.id === bugId);
    if (!bug) {
      toast({
        title: "Error",
        description: "Bug not found",
        variant: "destructive",
      });
      return false;
    }

    setFixing(bugId);

    try {
      toast({
        title: "Applying Fix",
        description: `Fixing: ${bug.errorMessage.substring(0, 50)}...`,
      });

      // Simulate fix application
      await new Promise(resolve => setTimeout(resolve, 2000));

      // Remove fixed bug from list
      setBugs(prev => prev.filter(b => b.id !== bugId));

      toast({
        title: "Fix Applied",
        description: "Bug fixed successfully",
      });

      return true;
    } catch (error: any) {
      toast({
        title: "Fix Failed",
        description: error.message,
        variant: "destructive",
      });
      return false;
    } finally {
      setFixing(null);
    }
  }, [bugs, toast]);

  const clearBugs = useCallback(() => {
    setBugs([]);
  }, []);

  const getBugsByFile = useCallback((filePath: string) => {
    return bugs.filter(bug => bug.affectedFiles.includes(filePath));
  }, [bugs]);

  const getCriticalBugs = useCallback(() => {
    return bugs.filter(bug => bug.severity === "critical");
  }, [bugs]);

  return {
    scanning,
    fixing,
    bugs,
    scanForErrors,
    applyAutoFix,
    clearBugs,
    getBugsByFile,
    getCriticalBugs,
  };
};
