import { useState, useCallback } from "react";
import { useToast } from "@/hooks/use-toast";

export interface ProjectDoc {
  id: string;
  projectName: string;
  version: string;
  generatedAt: Date;
  overview: string;
  modules: ModuleDoc[];
  routes: RouteDoc[];
  database: DatabaseDoc;
  apiEndpoints: APIEndpointDoc[];
}

export interface ModuleDoc {
  name: string;
  slug: string;
  description: string;
  features: string[];
  pages: string[];
  components: string[];
}

export interface RouteDoc {
  path: string;
  method?: string;
  component?: string;
  protected: boolean;
  description: string;
}

export interface DatabaseDoc {
  tables: TableDoc[];
  totalTables: number;
}

export interface TableDoc {
  name: string;
  columns: ColumnDoc[];
  description: string;
}

export interface ColumnDoc {
  name: string;
  type: string;
  nullable: boolean;
  description?: string;
}

export interface APIEndpointDoc {
  path: string;
  method: string;
  description: string;
  parameters?: string[];
  response?: string;
}

export const useAutoDocGenerator = () => {
  const [generating, setGenerating] = useState(false);
  const [docs, setDocs] = useState<ProjectDoc[]>([]);
  const { toast } = useToast();

  const generateDocs = useCallback(async (projectName: string): Promise<ProjectDoc | null> => {
    setGenerating(true);

    try {
      toast({
        title: "Generating Documentation",
        description: "Scanning project structure...",
      });

      // Simulate scanning
      await new Promise(resolve => setTimeout(resolve, 2000));

      const doc: ProjectDoc = {
        id: `doc_${Date.now()}`,
        projectName,
        version: "1.0.0",
        generatedAt: new Date(),
        overview: `Smart Friend ${projectName} is a comprehensive platform that provides advanced features and capabilities.`,
        modules: [
          {
            name: "Smart Friend Studio",
            slug: "studio",
            description: "Complete development environment with AI assistance",
            features: [
              "AI Code Assistant",
              "Module Builder",
              "DevOps Integration",
              "Deployment Tools",
            ],
            pages: ["/studio", "/studio/ide", "/studio/modules"],
            components: ["CodeEditor", "FileTree", "AIAssistant"],
          },
          {
            name: "Smart Marketer",
            slug: "marketer",
            description: "AI-powered marketing automation",
            features: [
              "Campaign Management",
              "Content Generation",
              "Analytics Dashboard",
            ],
            pages: ["/marketer", "/marketer/campaigns"],
            components: ["CampaignCard", "ContentEditor"],
          },
        ],
        routes: [
          {
            path: "/",
            component: "Landing",
            protected: false,
            description: "Landing page",
          },
          {
            path: "/studio",
            component: "Studio",
            protected: true,
            description: "Smart Friend Studio dashboard",
          },
          {
            path: "/studio/ide",
            component: "StudioIDE",
            protected: true,
            description: "Integrated Development Environment",
          },
        ],
        database: {
          totalTables: 15,
          tables: [
            {
              name: "profiles",
              description: "User profile information",
              columns: [
                { name: "id", type: "uuid", nullable: false, description: "Primary key" },
                { name: "full_name", type: "text", nullable: false, description: "User's full name" },
                { name: "phone", type: "text", nullable: true, description: "Phone number" },
                { name: "avatar_url", type: "text", nullable: true, description: "Avatar image URL" },
              ],
            },
            {
              name: "messages",
              description: "Chat messages",
              columns: [
                { name: "id", type: "uuid", nullable: false, description: "Primary key" },
                { name: "content", type: "text", nullable: false, description: "Message content" },
                { name: "role", type: "text", nullable: false, description: "Message role (user/assistant)" },
                { name: "user_id", type: "uuid", nullable: false, description: "User who sent message" },
              ],
            },
          ],
        },
        apiEndpoints: [
          {
            path: "/api/chat",
            method: "POST",
            description: "Send message to AI assistant",
            parameters: ["message", "context"],
            response: "AI response text",
          },
          {
            path: "/api/generate",
            method: "POST",
            description: "Generate code or content",
            parameters: ["prompt", "type"],
            response: "Generated content",
          },
        ],
      };

      setDocs(prev => [...prev, doc]);

      toast({
        title: "Documentation Generated",
        description: `Documentation for ${projectName} is ready`,
      });

      return doc;
    } catch (error: any) {
      toast({
        title: "Generation Failed",
        description: error.message,
        variant: "destructive",
      });
      return null;
    } finally {
      setGenerating(false);
    }
  }, [toast]);

  const getLatestDoc = useCallback(() => {
    if (docs.length === 0) return null;
    return docs[docs.length - 1];
  }, [docs]);

  const getDocById = useCallback((id: string) => {
    return docs.find(doc => doc.id === id);
  }, [docs]);

  const searchDocs = useCallback((query: string) => {
    const lowerQuery = query.toLowerCase();
    const results = docs.filter(doc => {
      return (
        doc.projectName.toLowerCase().includes(lowerQuery) ||
        doc.overview.toLowerCase().includes(lowerQuery) ||
        doc.modules.some(m => 
          m.name.toLowerCase().includes(lowerQuery) || 
          m.description.toLowerCase().includes(lowerQuery)
        )
      );
    });
    return results;
  }, [docs]);

  return {
    generating,
    docs,
    generateDocs,
    getLatestDoc,
    getDocById,
    searchDocs,
  };
};
