import { useState } from "react";
import { useToast } from "@/hooks/use-toast";

export type AvaCommandCategory = 
  | "study"
  | "fun"
  | "emotion"
  | "daily"
  | "chat"
  | "security"
  | "school";

export interface AvaCommand {
  category: AvaCommandCategory;
  action: string;
  subject?: string;
  emotion?: string;
  needsGuardian?: boolean;
  notifyOwner?: boolean;
}

export const useAvaCommandParser = () => {
  const { toast } = useToast();
  const [lastCommand, setLastCommand] = useState<AvaCommand | null>(null);

  const parseAvaCommand = (input: string): AvaCommand | null => {
    const lower = input.toLowerCase().trim();

    // Study mode commands
    if (lower === "study mode") {
      return {
        category: "study",
        action: "start_study_mode",
      };
    }

    if (lower.startsWith("teach me ")) {
      return {
        category: "study",
        action: "teach",
        subject: lower.replace("teach me ", ""),
      };
    }

    if (lower === "explain like i'm a child" || lower === "explain like i am a child") {
      return {
        category: "study",
        action: "simple_explain",
      };
    }

    if (lower === "ask me questions") {
      return {
        category: "study",
        action: "quiz_mode",
      };
    }

    if (lower === "check my homework") {
      return {
        category: "study",
        action: "check_homework",
      };
    }

    // Fun & friendship commands
    if (lower === "story mode") {
      return {
        category: "fun",
        action: "tell_story",
      };
    }

    if (lower === "play with me") {
      return {
        category: "fun",
        action: "play_game",
      };
    }

    if (lower === "sing a song" || lower === "tell me a joke" || 
        lower === "draw something" || lower === "make a cute picture") {
      return {
        category: "fun",
        action: lower.replace(/ /g, "_"),
      };
    }

    // Emotional + safety commands
    if (lower === "i am sad" || lower === "i am scared" || 
        lower === "i am angry" || lower === "i feel lonely") {
      const emotion = lower.replace("i am ", "").replace("i feel ", "");
      return {
        category: "emotion",
        action: "emotional_support",
        emotion,
        notifyOwner: emotion === "scared",
      };
    }

    if (lower === "check my safety" || lower === "stay with me" || lower === "protect me") {
      return {
        category: "security",
        action: "guardian_mode",
        needsGuardian: true,
      };
    }

    // Daily life commands
    if (lower === "make my routine" || lower === "what should i do now?" ||
        lower === "help me focus" || lower.startsWith("remind me to ")) {
      return {
        category: "daily",
        action: lower.startsWith("remind me to ") ? "set_reminder" : lower.replace(/ /g, "_"),
        subject: lower.startsWith("remind me to ") ? lower.replace("remind me to ", "") : undefined,
      };
    }

    // Chat & friendship commands
    if (lower === "talk to me" || lower === "be my friend" || 
        lower === "tell me something interesting" || lower === "let's chat" ||
        lower === "listen to me") {
      return {
        category: "chat",
        action: "friendly_chat",
      };
    }

    // Security + guardian commands
    if (lower === "report to baba") {
      return {
        category: "security",
        action: "report_to_owner",
        notifyOwner: true,
      };
    }

    if (lower === "i am in danger") {
      return {
        category: "security",
        action: "emergency_mode",
        needsGuardian: true,
        notifyOwner: true,
      };
    }

    if (lower === "where am i?") {
      return {
        category: "security",
        action: "location_info",
      };
    }

    // School & education commands
    if (lower === "explain my class" || lower === "teach like my school teacher" ||
        lower === "prepare me for exam" || lower === "make chapter summary" ||
        lower === "give me multiple choice questions") {
      return {
        category: "school",
        action: lower.replace(/ /g, "_"),
      };
    }

    return null;
  };

  const execute = (input: string) => {
    const parsed = parseAvaCommand(input);
    
    if (!parsed) {
      toast({
        title: "আমি বুঝতে পারিনি 😊",
        description: "তুমি কি আবার বলতে পারবে? আমি তোমাকে সাহায্য করতে চাই!",
      });
      return null;
    }

    setLastCommand(parsed);
    
    // Show friendly confirmation
    const friendlyMessages: Record<AvaCommandCategory, string> = {
      study: "চলো পড়াশোনা শুরু করি! 📚",
      fun: "মজা করার সময়! 🎉",
      emotion: "আমি তোমার সাথে আছি 💙",
      daily: "তোমার রুটিন তৈরি করছি ⏰",
      chat: "চলো গল্প করি! 😊",
      security: "আমি তোমাকে রক্ষা করছি 🛡️",
      school: "তোমার পড়া শেখাচ্ছি 🎓",
    };

    toast({
      title: friendlyMessages[parsed.category],
      description: parsed.needsGuardian ? "বাবাকে জানাচ্ছি..." : undefined,
    });

    return parsed;
  };

  return {
    parseAvaCommand,
    execute,
    lastCommand,
  };
};
