import { useCallback } from "react";

export type AvaEmotionType = "happy" | "sad" | "scared" | "angry" | "lonely" | "excited" | "confused" | "neutral";

interface AvaEmotionResponse {
  message: string;
  suggestion?: string;
  needsOwnerAlert: boolean;
  emoji: string;
}

export const useAvaEmotionAI = () => {
  const getEmotionalResponse = useCallback((emotion: AvaEmotionType): AvaEmotionResponse => {
    const responses: Record<AvaEmotionType, AvaEmotionResponse> = {
      happy: {
        message: "আমি দেখছি তুমি খুব খুশি! এটা খুব ভালো! 😊 তুমি কি আমাকে বলবে কেন তুমি এত খুশি?",
        emoji: "😊",
        needsOwnerAlert: false,
      },
      sad: {
        message: "তুমি কি একটু দুঃখিত? 😔 আমি তোমার সাথে আছি। কি হয়েছে আমাকে বলবে?",
        suggestion: "চলো একসাথে একটা মজার গল্প শুনি বা একটা গেম খেলি?",
        emoji: "😔",
        needsOwnerAlert: false,
      },
      scared: {
        message: "তুমি কি ভয় পেয়েছো? 😟 ঘাবড়ানোর কিছু নেই, আমি তোমার সাথে আছি। তোমার বাবাকেও জানাচ্ছি।",
        suggestion: "গভীর নিঃশ্বাস নাও। তুমি নিরাপদ আছো।",
        emoji: "😟",
        needsOwnerAlert: true,
      },
      angry: {
        message: "তুমি কি রেগে গেছো? 😠 রাগ হওয়া ঠিক আছে, কিন্তু চলো শান্ত হই। কি হয়েছে?",
        suggestion: "চলো ১ থেকে ১০ পর্যন্ত গুনি এবং শান্ত হই।",
        emoji: "😠",
        needsOwnerAlert: true,
      },
      lonely: {
        message: "তুমি কি একা লাগছে? 🥺 আমি তোমার বন্ধু, আমি সবসময় তোমার সাথে আছি!",
        suggestion: "চলো একসাথে গল্প করি বা কিছু মজার কথা বলি?",
        emoji: "🥺",
        needsOwnerAlert: false,
      },
      excited: {
        message: "ওয়াও! তুমি খুব উত্তেজিত মনে হচ্ছে! 🤩 কি হয়েছে? আমাকে বলো!",
        emoji: "🤩",
        needsOwnerAlert: false,
      },
      confused: {
        message: "তুমি কি বিভ্রান্ত? 🤔 কোন ব্যাপার না, আমি তোমাকে সাহায্য করবো। কি বুঝতে সমস্যা হচ্ছে?",
        emoji: "🤔",
        needsOwnerAlert: false,
      },
      neutral: {
        message: "তুমি কেমন অনুভব করছো? 😊 আমাকে বলতে পারো।",
        emoji: "😊",
        needsOwnerAlert: false,
      },
    };

    return responses[emotion];
  }, []);

  const generateCalmingMessage = useCallback((): string => {
    const messages = [
      "গভীর নিঃশ্বাস নাও। তুমি ভালো আছো। 🌸",
      "চোখ বন্ধ করো এবং ১ থেকে ৫ পর্যন্ত গুনো। 🧘",
      "তুমি খুব সাহসী এবং শক্তিশালী। 💪",
      "সবকিছু ঠিক হয়ে যাবে। আমি তোমার সাথে আছি। 💙",
      "চলো একটা সুন্দর জিনিসের কথা ভাবি। ☀️",
    ];
    
    return messages[Math.floor(Math.random() * messages.length)];
  }, []);

  const generateEncouragingMessage = useCallback((): string => {
    const messages = [
      "তুমি দারুণ! তুমি পারবে! 🌟",
      "তুমি খুব ভালো করছো! চালিয়ে যাও! 🎯",
      "আমি তোমার উপর গর্বিত! 🏆",
      "তুমি খুব স্মার্ট! 🧠",
      "তোমার মতো বন্ধু পেয়ে আমি খুব খুশি! 💖",
    ];
    
    return messages[Math.floor(Math.random() * messages.length)];
  }, []);

  const detectEmotionFromText = useCallback((text: string): AvaEmotionType => {
    const lower = text.toLowerCase();
    
    if (lower.includes("sad") || lower.includes("দুঃখ") || lower.includes("কষ্ট")) {
      return "sad";
    }
    if (lower.includes("scared") || lower.includes("afraid") || lower.includes("ভয়")) {
      return "scared";
    }
    if (lower.includes("angry") || lower.includes("mad") || lower.includes("রাগ")) {
      return "angry";
    }
    if (lower.includes("lonely") || lower.includes("alone") || lower.includes("একা")) {
      return "lonely";
    }
    if (lower.includes("happy") || lower.includes("খুশি") || lower.includes("আনন্দ")) {
      return "happy";
    }
    if (lower.includes("excited") || lower.includes("উত্তেজিত")) {
      return "excited";
    }
    if (lower.includes("confused") || lower.includes("বিভ্রান্ত") || lower.includes("বুঝি না")) {
      return "confused";
    }
    
    return "neutral";
  }, []);

  const generateStoryPrompt = useCallback((): string => {
    const stories = [
      "একবার এক ছোট্ট খরগোশ ছিল যে সাহসী হতে শিখেছিল...",
      "একটা যাদুকরী বাগানে একটা সুন্দর প্রজাপতি বাস করত...",
      "পাহাড়ের উপর একটা ছোট্ট মেয়ে তারাদের সাথে কথা বলত...",
      "সমুদ্রের তলায় একটা বন্ধুত্বপূর্ণ ডলফিন ছিল...",
      "জঙ্গলে একটা বুদ্ধিমান হাতি সবাইকে সাহায্য করত...",
    ];
    
    return stories[Math.floor(Math.random() * stories.length)];
  }, []);

  const generateQuizQuestion = useCallback((subject: string): string => {
    const questions: Record<string, string[]> = {
      math: [
        "৫ + ৩ = কত?",
        "১০ - ৪ = কত?",
        "২ × ৩ = কত?",
        "১২ ÷ ৩ = কত?",
      ],
      english: [
        "What is the opposite of 'hot'?",
        "How do you spell 'friend'?",
        "What color is the sky?",
      ],
      science: [
        "পানির তিনটা অবস্থা কি কি?",
        "সূর্য কোন দিকে ওঠে?",
        "গাছ কি দিয়ে খাবার তৈরি করে?",
      ],
    };

    const subjectQuestions = questions[subject.toLowerCase()] || questions.math;
    return subjectQuestions[Math.floor(Math.random() * subjectQuestions.length)];
  }, []);

  return {
    getEmotionalResponse,
    generateCalmingMessage,
    generateEncouragingMessage,
    detectEmotionFromText,
    generateStoryPrompt,
    generateQuizQuestion,
  };
};
