import { useState, useCallback } from "react";
import { useToast } from "@/hooks/use-toast";

export interface AvaSafetySettings {
  allowInternet: boolean;
  allowFileAccess: boolean;
  allowDeployment: boolean;
  allowDevOps: boolean;
  allowSystemFiles: boolean;
  allowAdultContent: boolean;
  maxSessionMinutes: number;
  requireOwnerApproval: boolean;
}

export interface AvaSession {
  startTime: Date;
  activities: string[];
  emotionalState: "happy" | "sad" | "scared" | "angry" | "neutral";
  lastInteraction: Date;
  needsOwnerAttention: boolean;
}

const DEFAULT_SAFETY_SETTINGS: AvaSafetySettings = {
  allowInternet: false,
  allowFileAccess: false,
  allowDeployment: false,
  allowDevOps: false,
  allowSystemFiles: false,
  allowAdultContent: false,
  maxSessionMinutes: 120,
  requireOwnerApproval: true,
};

export const useAvaMode = () => {
  const { toast } = useToast();
  const [isAvaMode, setIsAvaMode] = useState(false);
  const [safetySettings] = useState<AvaSafetySettings>(DEFAULT_SAFETY_SETTINGS);
  const [session, setSession] = useState<AvaSession | null>(null);

  const startAvaMode = useCallback(() => {
    setIsAvaMode(true);
    setSession({
      startTime: new Date(),
      activities: [],
      emotionalState: "neutral",
      lastInteraction: new Date(),
      needsOwnerAttention: false,
    });
    
    toast({
      title: "হাই আভা! 👋",
      description: "আমি তোমার বন্ধু SmartFriend! কেমন আছো?",
    });
  }, [toast]);

  const endAvaMode = useCallback(() => {
    if (session) {
      const duration = Math.floor((Date.now() - session.startTime.getTime()) / 60000);
      toast({
        title: "বিদায় আভা! 👋",
        description: `আজ আমরা ${duration} মিনিট একসাথে ছিলাম। আবার দেখা হবে!`,
      });
    }
    setIsAvaMode(false);
    setSession(null);
  }, [session, toast]);

  const logActivity = useCallback((activity: string) => {
    if (session) {
      setSession(prev => prev ? {
        ...prev,
        activities: [...prev.activities, activity],
        lastInteraction: new Date(),
      } : null);
    }
  }, [session]);

  const updateEmotionalState = useCallback((emotion: "happy" | "sad" | "scared" | "angry" | "neutral") => {
    if (session) {
      setSession(prev => prev ? {
        ...prev,
        emotionalState: emotion,
        needsOwnerAttention: emotion === "scared" || emotion === "angry",
      } : null);
      
      if (emotion === "scared" || emotion === "angry") {
        toast({
          title: "বাবাকে জানানো হচ্ছে...",
          description: "আমি তোমার বাবাকে খবর দিচ্ছি",
          variant: "destructive",
        });
      }
    }
  }, [session, toast]);

  const checkSafety = useCallback((action: string): boolean => {
    // Block unsafe actions
    const unsafeKeywords = [
      "deploy", "devops", "delete", "remove", "system", 
      "file", "admin", "root", "password", "config"
    ];
    
    const isUnsafe = unsafeKeywords.some(keyword => 
      action.toLowerCase().includes(keyword)
    );
    
    if (isUnsafe) {
      toast({
        title: "এটা আভার জন্য নিরাপদ নয়",
        description: "আমি তোমার বাবাকে জানাচ্ছি",
        variant: "destructive",
      });
      return false;
    }
    
    return true;
  }, [toast]);

  const notifyOwner = useCallback((reason: string, details: string) => {
    // Log to SmartFriend-1
    console.log("[AVA MODE] Owner notification:", { reason, details, session });
    
    toast({
      title: "বাবাকে জানানো হয়েছে",
      description: reason,
    });
  }, [session, toast]);

  return {
    isAvaMode,
    safetySettings,
    session,
    startAvaMode,
    endAvaMode,
    logActivity,
    updateEmotionalState,
    checkSafety,
    notifyOwner,
  };
};
