import { useState } from "react";
import { toast } from "sonner";

interface PipelineConfig {
  name: string;
  triggers: string[];
  jobs: string[];
}

interface Pipeline {
  name: string;
  type: "github-actions" | "gitlab-ci";
  content: string;
  jobs: string[];
  createdAt: string;
}

export const useCICDGenerator = () => {
  const [generating, setGenerating] = useState(false);
  const [pipelines, setPipelines] = useState<Pipeline[]>([]);

  const generateGitHubActions = async (config: PipelineConfig): Promise<Pipeline> => {
    setGenerating(true);
    toast.info("Generating GitHub Actions workflow...");
    
    await new Promise(resolve => setTimeout(resolve, 1500));
    
    const pipeline: Pipeline = {
      name: config.name,
      type: "github-actions",
      jobs: config.jobs,
      createdAt: new Date().toISOString(),
      content: `name: ${config.name}

on:
  ${config.triggers.map(t => `${t}:`).join('\n  ')}
    branches: [ main, develop ]

jobs:
  build:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v3
      
      - name: Setup Node.js
        uses: actions/setup-node@v3
        with:
          node-version: '18'
          cache: 'npm'
      
      - name: Install dependencies
        run: npm ci
      
      - name: Build
        run: npm run build
        env:
          VITE_SUPABASE_URL: \${{ secrets.VITE_SUPABASE_URL }}
          VITE_SUPABASE_PUBLISHABLE_KEY: \${{ secrets.VITE_SUPABASE_PUBLISHABLE_KEY }}
      
      - name: Run tests
        run: npm test

  test:
    runs-on: ubuntu-latest
    needs: build
    steps:
      - uses: actions/checkout@v3
      - run: npm ci
      - run: npm test

  deploy:
    runs-on: ubuntu-latest
    needs: [build, test]
    if: github.ref == 'refs/heads/main'
    steps:
      - uses: actions/checkout@v3
      - name: Deploy to production
        run: |
          echo "Deploying to production..."
          # Add deployment commands here
`
    };
    
    setPipelines(prev => [...prev, pipeline]);
    setGenerating(false);
    toast.success("GitHub Actions workflow generated");
    return pipeline;
  };

  const generateGitLabCI = async (config: PipelineConfig): Promise<Pipeline> => {
    setGenerating(true);
    toast.info("Generating GitLab CI pipeline...");
    
    await new Promise(resolve => setTimeout(resolve, 1500));
    
    const pipeline: Pipeline = {
      name: config.name,
      type: "gitlab-ci",
      jobs: config.jobs,
      createdAt: new Date().toISOString(),
      content: `stages:
  - build
  - test
  - deploy

variables:
  NODE_VERSION: "18"

build:
  stage: build
  image: node:\${NODE_VERSION}
  script:
    - npm ci
    - npm run build
  artifacts:
    paths:
      - dist/
    expire_in: 1 hour
  only:
    - main
    - develop

test:
  stage: test
  image: node:\${NODE_VERSION}
  script:
    - npm ci
    - npm test
  only:
    - main
    - develop

deploy:
  stage: deploy
  image: node:\${NODE_VERSION}
  script:
    - echo "Deploying to production..."
    - npm run deploy
  only:
    - main
  when: manual
`
    };
    
    setPipelines(prev => [...prev, pipeline]);
    setGenerating(false);
    toast.success("GitLab CI pipeline generated");
    return pipeline;
  };

  const exportPipeline = async (pipelineName: string): Promise<boolean> => {
    const pipeline = pipelines.find(p => p.name === pipelineName);
    if (!pipeline) {
      toast.error("Pipeline not found");
      return false;
    }

    const blob = new Blob([pipeline.content], { type: "text/yaml" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = pipeline.type === "github-actions" 
      ? ".github/workflows/main.yml" 
      : ".gitlab-ci.yml";
    a.click();
    URL.revokeObjectURL(url);

    toast.success("Pipeline configuration exported");
    return true;
  };

  return {
    generateGitHubActions,
    generateGitLabCI,
    exportPipeline,
    pipelines,
    generating
  };
};
