import { useState, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { toast } from "sonner";

export interface CallHistoryRecord {
  id: string;
  user_id: string;
  call_duration: number;
  transcript: string | null;
  ai_summary: string | null;
  created_at: string;
}

export const useCallHistory = () => {
  const [history, setHistory] = useState<CallHistoryRecord[]>([]);
  const [loading, setLoading] = useState(false);

  const fetchHistory = async () => {
    setLoading(true);
    try {
      const { data: { user } } = await supabase.auth.getUser();
      if (!user) throw new Error("Not authenticated");

      const { data, error } = await supabase
        .from("call_history")
        .select("*")
        .eq("user_id", user.id)
        .order("created_at", { ascending: false });

      if (error) throw error;

      setHistory(data || []);
    } catch (error: any) {
      console.error("Failed to fetch call history:", error);
      toast.error("Failed to load call history");
    } finally {
      setLoading(false);
    }
  };

  const deleteRecord = async (id: string) => {
    try {
      const { error } = await supabase
        .from("call_history")
        .delete()
        .eq("id", id);

      if (error) throw error;

      setHistory(prev => prev.filter(record => record.id !== id));
      toast.success("Record deleted");
    } catch (error: any) {
      console.error("Failed to delete record:", error);
      toast.error("Failed to delete record");
    }
  };

  const exportTranscript = (record: CallHistoryRecord) => {
    const content = `Call Summary\n\nDate: ${new Date(record.created_at).toLocaleString()}\nDuration: ${Math.floor(record.call_duration / 60)}m ${record.call_duration % 60}s\n\nSummary:\n${record.ai_summary || "N/A"}\n\nTranscript:\n${record.transcript || "No transcript available"}`;
    
    const blob = new Blob([content], { type: "text/plain" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `call-transcript-${record.id}.txt`;
    a.click();
    URL.revokeObjectURL(url);
    
    toast.success("Transcript exported");
  };

  useEffect(() => {
    fetchHistory();
  }, []);

  return {
    history,
    loading,
    fetchHistory,
    deleteRecord,
    exportTranscript
  };
};
