import { useState } from "react";
import { useToast } from "@/hooks/use-toast";
import { ParsedCommand } from "./useCommandParser";
import { useAgentManager } from "./useAgentManager";
import { useTaskEngine } from "./useTaskEngine";
import { useOwnerCheck } from "./useOwnerCheck";
import { useAvaCommandParser } from "./useAvaCommandParser";
import { useAvaMode } from "./useAvaMode";

export interface CommandExecution {
  id: string;
  command: ParsedCommand;
  status: "pending" | "running" | "completed" | "failed";
  startTime: Date;
  endTime?: Date;
  logs: string[];
  result?: any;
}

export const useCommandExecutor = () => {
  const { toast } = useToast();
  const { setAgentStatus, stopAll: stopAllAgents } = useAgentManager();
  const { createTask, startTask } = useTaskEngine();
  const { isOwner } = useOwnerCheck();
  const { execute: executeAvaCommand } = useAvaCommandParser();
  const { isAvaMode, checkSafety, notifyOwner } = useAvaMode();

  const [executions, setExecutions] = useState<CommandExecution[]>([]);
  const [currentExecution, setCurrentExecution] = useState<CommandExecution | null>(null);

  const addLog = (executionId: string, message: string) => {
    setExecutions(prev => prev.map(exec => 
      exec.id === executionId 
        ? { ...exec, logs: [...exec.logs, `[${new Date().toISOString()}] ${message}`] }
        : exec
    ));
  };

  const execute = async (command: ParsedCommand): Promise<void> => {
    // Check if this is an Ava mode command
    if (command.category === "child_mode") {
      const avaCommand = executeAvaCommand(command.params?.task || "");
      if (avaCommand) {
        toast({
          title: "Routed to SmartFriend-2 (Ava Mode)",
          description: "This task is being handled safely for Ava",
        });
      }
      return;
    }

    // Safety check for Ava mode
    if (isAvaMode && !checkSafety(command.action)) {
      notifyOwner(
        "Unsafe action blocked",
        `Ava tried to: ${command.action}`
      );
      return;
    }

    // Check owner authentication for sensitive commands
    if (command.requiresOwnerAuth && !isOwner) {
      toast({
        title: "Owner authentication required",
        description: "This command can only be executed by the owner",
        variant: "destructive",
      });
      return;
    }

    const execution: CommandExecution = {
      id: Math.random().toString(36).substring(7),
      command,
      status: "pending",
      startTime: new Date(),
      logs: [],
    };

    setExecutions(prev => [...prev, execution]);
    setCurrentExecution(execution);

    try {
      execution.status = "running";
      addLog(execution.id, `Starting command: ${command.action}`);
      addLog(execution.id, `Routing to agents: ${command.agents.join(", ")}`);

      // Set agents to busy
      command.agents.forEach(agent => {
        setAgentStatus(agent.toLowerCase().replace("agent", ""), "busy", command.action);
      });

      // Execute based on category
      switch (command.category) {
        case "owner_override":
          await executeOwnerOverride(execution);
          break;
        case "global":
          await executeGlobal(execution);
          break;
        case "module":
          await executeModule(execution);
          break;
        case "development":
          await executeDevelopment(execution);
          break;
        case "marketing":
          await executeMarketing(execution);
          break;
        case "media":
          await executeMedia(execution);
          break;
        case "security":
          await executeSecurity(execution);
          break;
        case "teamwork":
          await executeTeamwork(execution);
          break;
        default:
          await executeGeneric(execution);
      }

      execution.status = "completed";
      execution.endTime = new Date();
      addLog(execution.id, "Command completed successfully");

      toast({
        title: "Command completed",
        description: `${command.action} finished successfully`,
      });
    } catch (error) {
      execution.status = "failed";
      execution.endTime = new Date();
      addLog(execution.id, `Error: ${error instanceof Error ? error.message : "Unknown error"}`);

      toast({
        title: "Command failed",
        description: error instanceof Error ? error.message : "Unknown error",
        variant: "destructive",
      });
    } finally {
      // Set agents back to idle
      command.agents.forEach(agent => {
        setAgentStatus(agent.toLowerCase().replace("agent", ""), "idle");
      });
    }
  };

  const executeOwnerOverride = async (execution: CommandExecution) => {
    const { action, target } = execution.command;
    
    addLog(execution.id, "Owner override command detected");
    
    switch (action) {
      case "stop_all":
        addLog(execution.id, "Stopping all agents...");
        stopAllAgents();
        break;
      case "emergency_override":
        addLog(execution.id, "Entering emergency override mode...");
        // Implement emergency mode logic
        break;
      case "system_reset":
        addLog(execution.id, "Resetting agent states...");
        // Implement reset logic
        break;
      case "disable_agent":
        addLog(execution.id, `Disabling agent: ${target}`);
        // Implement via useAgentManager
        break;
      case "enable_agent":
        addLog(execution.id, `Enabling agent: ${target}`);
        // Implement via useAgentManager
        break;
      case "clone_backup":
        addLog(execution.id, "Creating system backup...");
        // Implement backup logic
        break;
    }
  };

  const executeGlobal = async (execution: CommandExecution) => {
    const { action, target, params } = execution.command;
    
    switch (action) {
      case "execute":
        addLog(execution.id, `Auto-routing task: ${target}`);
        // AI decides best agent
        break;
      case "assign":
        addLog(execution.id, `Assigning to ${target}: ${params?.task}`);
        // Direct assignment
        break;
      case "summary":
      case "status_report":
        addLog(execution.id, "Generating status report...");
        // Get status from all agents
        break;
      case "fix_everything":
        addLog(execution.id, "Triggering system-wide scan and fix...");
        const fixTask = createTask({
          title: "System-wide Fix",
          description: "Scan entire system for errors and apply fixes",
          type: "fix_bug",
          target: "all",
          priority: "high",
        });
        await startTask(fixTask.id);
        break;
      case "optimize_system":
        addLog(execution.id, "Optimizing system...");
        // Clean logs, rebuild indexes, etc.
        break;
    }
  };

  const executeModule = async (execution: CommandExecution) => {
    const { action, target } = execution.command;
    
    addLog(execution.id, `Module operation: ${action} on ${target}`);
    
    const task = createTask({
      title: `${action.replace("_", " ")} ${target}`,
      description: `Automated ${action} for ${target} module`,
      type: action === "build_module" ? "build_module" : 
            action === "fix_module" ? "fix_bug" : "optimize",
      target: target || "",
      priority: "medium",
    });
    
    await startTask(task.id);
    addLog(execution.id, `Task created: ${task.id}`);
  };

  const executeDevelopment = async (execution: CommandExecution) => {
    const { action, target } = execution.command;
    addLog(execution.id, `Development task: ${action} ${target || ""}`);
    
    const task = createTask({
      title: `${action.replace("_", " ")} ${target || ""}`,
      description: `Automated development task`,
      type: "build_module",
      target: target || "",
      priority: "medium",
    });
    
    await startTask(task.id);
  };

  const executeMarketing = async (execution: CommandExecution) => {
    const { action, target, params } = execution.command;
    addLog(execution.id, `Marketing task: ${action}`);
    
    // Simulate marketing operations
    await new Promise(resolve => setTimeout(resolve, 2000));
    addLog(execution.id, `Generated ${action} for ${target || params?.description || "campaign"}`);
  };

  const executeMedia = async (execution: CommandExecution) => {
    const { action, target } = execution.command;
    addLog(execution.id, `Media task: ${action}`);
    
    // Simulate media operations
    await new Promise(resolve => setTimeout(resolve, 2000));
    addLog(execution.id, `Completed ${action} ${target ? `for ${target}` : ""}`);
  };

  const executeSecurity = async (execution: CommandExecution) => {
    const { action } = execution.command;
    addLog(execution.id, `Security task: ${action}`);
    
    // Integrate with existing security/devops systems
    await new Promise(resolve => setTimeout(resolve, 2000));
    addLog(execution.id, `Security check completed`);
  };

  const executeTeamwork = async (execution: CommandExecution) => {
    const { params } = execution.command;
    const steps = params?.steps || [];
    
    addLog(execution.id, `Chain execution with ${steps.length} steps`);
    
    for (let i = 0; i < steps.length; i++) {
      addLog(execution.id, `Step ${i + 1}/${steps.length}: ${steps[i]}`);
      await new Promise(resolve => setTimeout(resolve, 1500));
    }
    
    addLog(execution.id, "All steps completed");
  };

  const executeGeneric = async (execution: CommandExecution) => {
    addLog(execution.id, "Executing generic command...");
    await new Promise(resolve => setTimeout(resolve, 1000));
    addLog(execution.id, "Generic command completed");
  };

  return {
    execute,
    executions,
    currentExecution,
  };
};
