import { useState } from "react";
import { useToast } from "@/hooks/use-toast";

export type CommandCategory = 
  | "global"
  | "child_mode"
  | "module"
  | "development"
  | "marketing"
  | "media"
  | "security"
  | "business"
  | "legal_medical"
  | "travel"
  | "teamwork"
  | "owner_override";

export interface ParsedCommand {
  category: CommandCategory;
  action: string;
  target?: string;
  params?: Record<string, any>;
  requiresOwnerAuth?: boolean;
  agents: string[];
}

export const useCommandParser = () => {
  const { toast } = useToast();
  const [lastCommand, setLastCommand] = useState<ParsedCommand | null>(null);

  const parseCommand = (input: string): ParsedCommand | null => {
    const lower = input.toLowerCase().trim();

    // Owner override commands
    if (lower === "emergency override" || lower === "full system reset" || 
        lower.startsWith("disable agent") || lower.startsWith("enable agent") ||
        lower === "make clone backup" || lower === "stop all") {
      return {
        category: "owner_override",
        action: lower.includes("override") ? "emergency_override" : 
                lower.includes("reset") ? "system_reset" :
                lower.includes("disable") ? "disable_agent" :
                lower.includes("enable") ? "enable_agent" :
                lower.includes("backup") ? "clone_backup" : "stop_all",
        target: lower.includes("agent") ? lower.split("agent ")[1] : undefined,
        requiresOwnerAuth: true,
        agents: ["SmartFriend1"],
      };
    }

    // Global commands
    if (lower.startsWith("execute ")) {
      return {
        category: "global",
        action: "execute",
        target: lower.replace("execute ", ""),
        agents: ["SmartFriend1"],
      };
    }

    if (lower.startsWith("assign to agent ")) {
      const parts = lower.split(": ");
      const agentName = parts[0].replace("assign to agent ", "");
      return {
        category: "global",
        action: "assign",
        target: agentName,
        params: { task: parts[1] },
        agents: [agentName],
      };
    }

    if (lower === "summary" || lower === "status report") {
      return {
        category: "global",
        action: lower.replace(" ", "_"),
        agents: ["SmartFriend1"],
      };
    }

    if (lower === "fix everything") {
      return {
        category: "global",
        action: "fix_everything",
        agents: ["BugFixAgent", "AutoDevOpsAgent"],
      };
    }

    if (lower === "optimize system") {
      return {
        category: "global",
        action: "optimize_system",
        agents: ["AutoDevOpsAgent"],
      };
    }

    if (lower.startsWith("open workspace ")) {
      return {
        category: "global",
        action: "open_workspace",
        target: lower.replace("open workspace ", ""),
        agents: ["SmartFriend1"],
      };
    }

    // Child mode commands
    if (lower.startsWith("for ava:") || lower.startsWith("send to ava assistant:")) {
      const task = lower.includes("for ava:") 
        ? lower.split("for ava:")[1].trim()
        : lower.split("send to ava assistant:")[1].trim();
      return {
        category: "child_mode",
        action: "route_to_ava",
        params: { task },
        agents: ["SmartFriend2"],
      };
    }

    // Module building
    if (lower.startsWith("build module ")) {
      return {
        category: "module",
        action: "build_module",
        target: lower.replace("build module ", ""),
        agents: ["WebDeveloperAgent", "LaravelExpertAgent", "AutoDevOpsAgent"],
      };
    }

    if (lower.startsWith("update module ")) {
      return {
        category: "module",
        action: "update_module",
        target: lower.replace("update module ", ""),
        agents: ["WebDeveloperAgent", "LaravelExpertAgent"],
      };
    }

    if (lower.startsWith("fix module ")) {
      return {
        category: "module",
        action: "fix_module",
        target: lower.replace("fix module ", ""),
        agents: ["BugFixAgent", "WebDeveloperAgent"],
      };
    }

    // Development commands
    if (lower.startsWith("generate backend for ")) {
      return {
        category: "development",
        action: "generate_backend",
        target: lower.replace("generate backend for ", ""),
        agents: ["LaravelExpertAgent"],
      };
    }

    if (lower.startsWith("generate frontend for ")) {
      return {
        category: "development",
        action: "generate_frontend",
        target: lower.replace("generate frontend for ", ""),
        agents: ["WebDeveloperAgent"],
      };
    }

    if (lower.startsWith("create api for ")) {
      return {
        category: "development",
        action: "create_api",
        target: lower.replace("create api for ", ""),
        agents: ["LaravelExpertAgent"],
      };
    }

    if (lower.startsWith("fix api ")) {
      return {
        category: "development",
        action: "fix_api",
        target: lower.replace("fix api ", ""),
        agents: ["BugFixAgent", "LaravelExpertAgent"],
      };
    }

    if (lower === "review code" || lower === "explain code") {
      return {
        category: "development",
        action: lower.replace(" ", "_"),
        agents: ["WebDeveloperAgent"],
      };
    }

    // Marketing & content
    if (lower.startsWith("write content: ")) {
      return {
        category: "marketing",
        action: "write_content",
        target: lower.replace("write content: ", ""),
        agents: ["ContentWriterAgent"],
      };
    }

    if (lower.startsWith("write news: ")) {
      return {
        category: "marketing",
        action: "write_news",
        target: lower.replace("write news: ", ""),
        agents: ["NewsWriterAgent"],
      };
    }

    if (lower.startsWith("create ad: ")) {
      return {
        category: "marketing",
        action: "create_ad",
        target: lower.replace("create ad: ", ""),
        agents: ["SmartMarketerAgent", "DesignerAgent"],
      };
    }

    if (lower === "create 30 social posts") {
      return {
        category: "marketing",
        action: "create_social_posts",
        params: { count: 30 },
        agents: ["ContentWriterAgent", "SmartMarketerAgent"],
      };
    }

    if (lower.startsWith("make seo plan for ")) {
      return {
        category: "marketing",
        action: "seo_plan",
        target: lower.replace("make seo plan for ", ""),
        agents: ["SEOAssistantAgent"],
      };
    }

    if (lower.startsWith("design banner: ")) {
      return {
        category: "marketing",
        action: "design_banner",
        target: lower.replace("design banner: ", ""),
        agents: ["DesignerAgent"],
      };
    }

    if (lower.startsWith("create logo: ")) {
      return {
        category: "marketing",
        action: "create_logo",
        target: lower.replace("create logo: ", ""),
        agents: ["LogoCreatorAgent"],
      };
    }

    // File & media
    if (lower.startsWith("make pdf: ")) {
      return {
        category: "media",
        action: "make_pdf",
        target: lower.replace("make pdf: ", ""),
        agents: ["PDFMakerAgent"],
      };
    }

    if (lower.startsWith("edit photo: ")) {
      return {
        category: "media",
        action: "edit_photo",
        target: lower.replace("edit photo: ", ""),
        agents: ["PhotoEditorAgent"],
      };
    }

    if (lower.startsWith("edit video: ")) {
      return {
        category: "media",
        action: "edit_video",
        target: lower.replace("edit video: ", ""),
        agents: ["VideoEditorAgent"],
      };
    }

    if (lower === "create promo video") {
      return {
        category: "media",
        action: "create_promo_video",
        agents: ["VideoEditorAgent", "ScriptWriterAgent"],
      };
    }

    if (lower === "remove background from image") {
      return {
        category: "media",
        action: "remove_background",
        agents: ["PhotoEditorAgent"],
      };
    }

    if (lower === "enhance image quality") {
      return {
        category: "media",
        action: "enhance_image",
        agents: ["PhotoEditorAgent"],
      };
    }

    // Security
    if (lower === "check system security") {
      return {
        category: "security",
        action: "security_check",
        agents: ["AutoDevOpsAgent", "BugFixAgent"],
      };
    }

    if (lower === "scan system errors") {
      return {
        category: "security",
        action: "scan_errors",
        agents: ["BugFixAgent"],
      };
    }

    if (lower === "check hosting health") {
      return {
        category: "security",
        action: "hosting_health",
        agents: ["AutoDevOpsAgent"],
      };
    }

    if (lower === "optimize hosting") {
      return {
        category: "security",
        action: "optimize_hosting",
        agents: ["AutoDevOpsAgent"],
      };
    }

    if (lower === "deploy now") {
      return {
        category: "security",
        action: "deploy_now",
        requiresOwnerAuth: true,
        agents: ["AutoDevOpsAgent"],
      };
    }

    if (lower === "rollback deployment") {
      return {
        category: "security",
        action: "rollback",
        requiresOwnerAuth: true,
        agents: ["AutoDevOpsAgent"],
      };
    }

    // Business
    if (lower === "create invoice" || lower === "calculate profit" || 
        lower === "business report" || lower === "create ledger" || 
        lower === "prepare tax file") {
      return {
        category: "business",
        action: lower.replace(/ /g, "_"),
        agents: ["AccountantAgent"],
      };
    }

    // Legal & Medical
    if (lower === "prepare legal notice" || lower === "write agreement") {
      return {
        category: "legal_medical",
        action: lower.replace(/ /g, "_"),
        agents: ["LawyerDraftAgent"],
      };
    }

    if (lower === "analyze medical report" || lower === "prepare doctor summary") {
      return {
        category: "legal_medical",
        action: lower.replace(/ /g, "_"),
        agents: ["DoctorReportAgent"],
      };
    }

    // Travel
    if (lower === "make tourist plan" || lower === "analyze city" || 
        lower === "plan hotel booking") {
      return {
        category: "travel",
        action: lower.replace(/ /g, "_"),
        agents: ["TouristInfoAgent", "HospitalReceptionAgent"],
      };
    }

    // Teamwork (chain execution)
    if (lower.includes("→") || lower.includes("->")) {
      const steps = lower.split(/→|->/).map(s => s.trim());
      return {
        category: "teamwork",
        action: "chain_execution",
        params: { steps },
        agents: steps.map(step => {
          if (step.includes("script")) return "ScriptWriterAgent";
          if (step.includes("video")) return "VideoEditorAgent";
          if (step.includes("banner") || step.includes("design")) return "DesignerAgent";
          if (step.includes("marketing")) return "SmartMarketerAgent";
          if (step.includes("frontend")) return "WebDeveloperAgent";
          if (step.includes("backend")) return "LaravelExpertAgent";
          if (step.includes("deploy")) return "AutoDevOpsAgent";
          return "SmartFriend1";
        }),
      };
    }

    // Shorthand chain commands
    if (lower.startsWith("create promo:")) {
      return {
        category: "teamwork",
        action: "chain_execution",
        params: { 
          steps: ["script", "video", "banner", "marketing"],
          description: lower.replace("create promo:", "").trim()
        },
        agents: ["ScriptWriterAgent", "VideoEditorAgent", "DesignerAgent", "SmartMarketerAgent"],
      };
    }

    if (lower.startsWith("create website:")) {
      return {
        category: "teamwork",
        action: "chain_execution",
        params: { 
          steps: ["design", "frontend", "backend", "deploy"],
          description: lower.replace("create website:", "").trim()
        },
        agents: ["DesignerAgent", "WebDeveloperAgent", "LaravelExpertAgent", "AutoDevOpsAgent"],
      };
    }

    return null;
  };

  const execute = (input: string) => {
    const parsed = parseCommand(input);
    
    if (!parsed) {
      toast({
        title: "Command not recognized",
        description: "Please check your command syntax",
        variant: "destructive",
      });
      return null;
    }

    setLastCommand(parsed);
    
    toast({
      title: `Command parsed: ${parsed.action}`,
      description: `Routing to: ${parsed.agents.join(", ")}`,
    });

    return parsed;
  };

  return {
    parseCommand,
    execute,
    lastCommand,
  };
};
