import { useState } from "react";
import { toast } from "sonner";

export interface ComponentAnalysis {
  name: string;
  type: "layout" | "section" | "ui";
  html: string;
  jsx: string;
  props: string[];
  dependencies: string[];
}

export const useComponentBuilder = () => {
  const [building, setBuilding] = useState(false);
  const [components, setComponents] = useState<ComponentAnalysis[]>([]);

  const analyzeHTML = (html: string): ComponentAnalysis[] => {
    const detected: ComponentAnalysis[] = [];

    // Detect common patterns
    if (html.includes("nav") || html.includes("menu")) {
      detected.push({
        name: "Header",
        type: "layout",
        html: "",
        jsx: "// Header component",
        props: [],
        dependencies: [],
      });
    }

    if (html.includes("footer")) {
      detected.push({
        name: "Footer",
        type: "layout",
        html: "",
        jsx: "// Footer component",
        props: [],
        dependencies: [],
      });
    }

    if (html.includes("hero") || html.includes("banner")) {
      detected.push({
        name: "Hero",
        type: "section",
        html: "",
        jsx: "// Hero section",
        props: [],
        dependencies: [],
      });
    }

    return detected;
  };

  const convertHTMLToJSX = (html: string): string => {
    let jsx = html;

    // Convert class to className
    jsx = jsx.replace(/class=/g, "className=");
    
    // Convert inline styles
    jsx = jsx.replace(/style="([^"]*)"/g, (_, styles) => {
      const styleObj = styles
        .split(";")
        .filter(Boolean)
        .map((s: string) => {
          const [key, val] = s.split(":").map((x: string) => x.trim());
          const camelKey = key.replace(/-./g, (x: string) => x[1].toUpperCase());
          return `${camelKey}: "${val}"`;
        })
        .join(", ");
      return `style={{${styleObj}}}`;
    });

    // Convert self-closing tags
    jsx = jsx.replace(/<(img|input|br|hr)([^>]*)>/g, "<$1$2 />");

    return jsx;
  };

  const buildComponents = async (html: string, useTailwind: boolean = true) => {
    setBuilding(true);
    toast.info("Building components...");

    try {
      await new Promise(resolve => setTimeout(resolve, 1500));

      const analyzed = analyzeHTML(html);
      const converted = analyzed.map(comp => ({
        ...comp,
        jsx: convertHTMLToJSX(comp.html || ""),
      }));

      setComponents(converted);
      toast.success(`Built ${converted.length} components`);
      return converted;

    } catch (error: any) {
      toast.error(`Failed to build components: ${error.message}`);
      return [];
    } finally {
      setBuilding(false);
    }
  };

  const generateComponentFile = (component: ComponentAnalysis): string => {
    return `import React from 'react';

export interface ${component.name}Props {
${component.props.map(prop => `  ${prop}: string;`).join("\n")}
}

export const ${component.name}: React.FC<${component.name}Props> = (props) => {
  return (
    <div className="container">
      {/* ${component.name} content */}
      ${component.jsx}
    </div>
  );
};
`;
  };

  return {
    buildComponents,
    analyzeHTML,
    convertHTMLToJSX,
    generateComponentFile,
    building,
    components,
  };
};
