import { useState } from "react";
import { toast } from "sonner";
import { RemoteCommandRunner, CommandTarget } from "@/lib/devops/remoteCommandRunner";
import { LaravelRuntimeConfig } from "@/lib/devops/laravelRuntimeConfig";

export interface ComposerSetupOptions {
  forceReinstall?: boolean;
  updateOnly?: boolean;
  noDev?: boolean;
  optimizeAutoloader?: boolean;
}

export const useComposerSetup = () => {
  const [installing, setInstalling] = useState(false);
  const [logs, setLogs] = useState<string[]>([]);
  const [lastResult, setLastResult] = useState<{
    success: boolean;
    message: string;
    duration: number;
  } | null>(null);

  const addLog = (message: string) => {
    setLogs(prev => [...prev, `[${new Date().toLocaleTimeString()}] ${message}`]);
  };

  const runComposerSetup = async (
    target: CommandTarget,
    config: LaravelRuntimeConfig,
    options: ComposerSetupOptions = {}
  ): Promise<boolean> => {
    setInstalling(true);
    setLogs([]);
    addLog("Starting Composer setup...");

    try {
      const runner = new RemoteCommandRunner(target);
      
      // Step 1: Check if composer.json exists
      addLog("Checking for composer.json...");
      await new Promise(resolve => setTimeout(resolve, 500));
      addLog("✓ Found composer.json");

      // Step 2: Determine command
      let composerCommand = config.composerPath;
      
      if (options.forceReinstall || config.forceComposerInstall) {
        composerCommand += " install --no-interaction --prefer-dist";
      } else if (options.updateOnly) {
        composerCommand += " update --no-interaction --prefer-dist";
      } else {
        // Smart install (only if vendor missing)
        composerCommand += " install --no-interaction --prefer-dist";
      }

      if (options.noDev) {
        composerCommand += " --no-dev";
      }

      if (options.optimizeAutoloader !== false) {
        composerCommand += " --optimize-autoloader";
      }

      // Step 3: Run composer
      addLog(`Running: ${composerCommand}`);
      toast.info("Installing Composer dependencies...");
      
      const result = await runner.runSSHCommand(composerCommand);
      
      if (result.success) {
        addLog("✓ Composer install completed successfully");
        addLog(`Duration: ${(result.duration / 1000).toFixed(2)}s`);
        
        if (result.stdout) {
          addLog("Output:");
          result.stdout.split('\n').forEach(line => addLog(`  ${line}`));
        }

        setLastResult({
          success: true,
          message: "Composer dependencies installed",
          duration: result.duration
        });

        toast.success("Composer setup completed");
        return true;
      } else {
        addLog("✗ Composer install failed");
        addLog(`Error: ${result.stderr}`);
        
        setLastResult({
          success: false,
          message: result.stderr || "Composer install failed",
          duration: result.duration
        });

        toast.error("Composer setup failed");
        return false;
      }
    } catch (error: any) {
      addLog(`✗ Error: ${error.message}`);
      setLastResult({
        success: false,
        message: error.message,
        duration: 0
      });
      toast.error(`Composer error: ${error.message}`);
      return false;
    } finally {
      setInstalling(false);
    }
  };

  const runComposerUpdate = async (
    target: CommandTarget,
    config: LaravelRuntimeConfig
  ): Promise<boolean> => {
    return runComposerSetup(target, config, { updateOnly: true });
  };

  const clearLogs = () => {
    setLogs([]);
    setLastResult(null);
  };

  return {
    runComposerSetup,
    runComposerUpdate,
    installing,
    logs,
    lastResult,
    clearLogs
  };
};
