import { useState, useCallback } from "react";

interface ConversationContext {
  summary: string;
  keyPoints: string[];
  entities: string[];
  timestamp: Date;
}

export const useContextRegeneration = () => {
  const [contexts, setContexts] = useState<ConversationContext[]>([]);

  // Compress conversation to summary
  const compressConversation = useCallback((messages: any[]): string => {
    if (messages.length === 0) return "";

    // Extract key information
    const userMessages = messages.filter(m => m.role === "user");
    const topics = new Set<string>();
    const entities = new Set<string>();

    userMessages.forEach(msg => {
      const content = msg.content.toLowerCase();
      
      // Extract topics (simplified)
      if (content.includes("গণিত") || content.includes("math")) topics.add("math");
      if (content.includes("বিজ্ঞান") || content.includes("science")) topics.add("science");
      if (content.includes("আইসিটি") || content.includes("ict")) topics.add("ict");
      if (content.includes("ইতিহাস") || content.includes("history")) topics.add("history");
      
      // Extract entities (names, places, etc.)
      const words = content.split(" ");
      words.forEach(word => {
        if (word.length > 3 && /^[A-Z]/.test(word)) {
          entities.add(word);
        }
      });
    });

    const summary = `Conversation about: ${Array.from(topics).join(", ")}. ${
      entities.size > 0 ? `Mentioned: ${Array.from(entities).join(", ")}.` : ""
    } ${messages.length} messages exchanged.`;

    return summary;
  }, []);

  // Generate context from messages
  const generateContext = useCallback((messages: any[]): ConversationContext => {
    const summary = compressConversation(messages);
    const keyPoints = messages
      .filter(m => m.role === "user")
      .slice(-5)
      .map(m => m.content.slice(0, 100));

    const entities: string[] = [];
    messages.forEach(msg => {
      const words = msg.content.split(" ");
      words.forEach(word => {
        if (word.length > 3 && /^[A-Z]/.test(word) && !entities.includes(word)) {
          entities.push(word);
        }
      });
    });

    return {
      summary,
      keyPoints,
      entities: entities.slice(0, 10),
      timestamp: new Date(),
    };
  }, [compressConversation]);

  // Save context
  const saveContext = useCallback((messages: any[]) => {
    const context = generateContext(messages);
    setContexts(prev => [...prev.slice(-9), context]);
    
    try {
      localStorage.setItem("conversation_contexts", JSON.stringify([...contexts.slice(-9), context]));
    } catch (error) {
      console.error("Failed to save context:", error);
    }
  }, [contexts, generateContext]);

  // Regenerate full context
  const regenerateFullContext = useCallback((): string => {
    if (contexts.length === 0) return "";

    const summaries = contexts.map(c => c.summary).join("\n");
    const allEntities = [...new Set(contexts.flatMap(c => c.entities))];
    
    return `Previous conversation summaries:\n${summaries}\n\nKey entities mentioned: ${allEntities.join(", ")}`;
  }, [contexts]);

  // Load contexts from storage
  const loadContexts = useCallback(() => {
    try {
      const stored = localStorage.getItem("conversation_contexts");
      if (stored) {
        const parsed = JSON.parse(stored);
        setContexts(parsed.map((c: any) => ({
          ...c,
          timestamp: new Date(c.timestamp),
        })));
      }
    } catch (error) {
      console.error("Failed to load contexts:", error);
    }
  }, []);

  // Clear all contexts
  const clearContexts = useCallback(() => {
    setContexts([]);
    localStorage.removeItem("conversation_contexts");
  }, []);

  return {
    contexts,
    saveContext,
    regenerateFullContext,
    loadContexts,
    clearContexts,
  };
};