import { useState } from "react";
import { devBrainEngine, DevBrainAnalysis } from "@/core/devBrain/DevBrainEngine";
import { DevBrainHistory, DevBrainLog } from "@/core/devBrain/DevBrainHistory";
import { useToast } from "@/hooks/use-toast";

export const useDevBrain = () => {
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [analysis, setAnalysis] = useState<DevBrainAnalysis | null>(null);
  const [diff, setDiff] = useState<string | null>(null);
  const [newCode, setNewCode] = useState<string | null>(null);
  const [logs, setLogs] = useState<DevBrainLog[]>([]);
  const { toast } = useToast();
  const history = new DevBrainHistory();

  const analyzeFile = async (filePath: string, content: string) => {
    setIsAnalyzing(true);
    try {
      const result = await devBrainEngine.analyzeFile(filePath, content);
      setAnalysis(result);
      toast({ title: "Analysis complete", description: "Code analysis finished successfully" });
      return result;
    } catch (error: any) {
      toast({
        title: "Analysis failed",
        description: error.message,
        variant: "destructive",
      });
      return null;
    } finally {
      setIsAnalyzing(false);
    }
  };

  const analyzeError = async (errorLog: string) => {
    setIsAnalyzing(true);
    try {
      const result = await devBrainEngine.analyzeError(errorLog);
      setAnalysis(result);
      toast({ title: "Error analysis complete" });
      return result;
    } catch (error: any) {
      toast({
        title: "Analysis failed",
        description: error.message,
        variant: "destructive",
      });
      return null;
    } finally {
      setIsAnalyzing(false);
    }
  };

  const suggestRefactor = async (params: {
    filePath: string;
    selectedCode: string;
    goal: string;
  }) => {
    setIsAnalyzing(true);
    try {
      const result = await devBrainEngine.suggestRefactor(params);
      setDiff(result.diff);
      setNewCode(result.newCode);
      toast({ title: "Refactor suggestion ready", description: "Review the changes below" });
      return result;
    } catch (error: any) {
      toast({
        title: "Refactor failed",
        description: error.message,
        variant: "destructive",
      });
      return null;
    } finally {
      setIsAnalyzing(false);
    }
  };

  const generateUpgradePlan = async () => {
    setIsAnalyzing(true);
    try {
      const result = await devBrainEngine.generateUpgradePlan();
      setAnalysis(result);
      toast({ title: "Upgrade plan generated" });
      return result;
    } catch (error: any) {
      toast({
        title: "Generation failed",
        description: error.message,
        variant: "destructive",
      });
      return null;
    } finally {
      setIsAnalyzing(false);
    }
  };

  const applyChanges = async (filePath: string, code: string, diffContent: string) => {
    try {
      await devBrainEngine.applyChanges(filePath, code, diffContent);
      toast({
        title: "Changes applied",
        description: "Code has been updated successfully",
      });
      setDiff(null);
      setNewCode(null);
      await loadLogs();
    } catch (error: any) {
      toast({
        title: "Apply failed",
        description: error.message,
        variant: "destructive",
      });
    }
  };

  const loadLogs = async (limit?: number) => {
    try {
      const recentLogs = await history.getRecentLogs(limit);
      setLogs(recentLogs);
    } catch (error) {
      console.error("Error loading logs:", error);
    }
  };

  const clearAnalysis = () => {
    setAnalysis(null);
    setDiff(null);
    setNewCode(null);
  };

  return {
    analyzeFile,
    analyzeError,
    suggestRefactor,
    generateUpgradePlan,
    applyChanges,
    loadLogs,
    clearAnalysis,
    isAnalyzing,
    analysis,
    diff,
    newCode,
    logs,
  };
};
