import { useState } from "react";
import { toast } from "sonner";

interface DockerFile {
  name: string;
  content: string;
  type: "dockerfile" | "compose" | "ignore";
}

export const useDockerGenerator = () => {
  const [generating, setGenerating] = useState(false);
  const [dockerFiles, setDockerFiles] = useState<DockerFile[]>([]);

  const generateDockerfile = async (
    baseImage: string = "node:18-alpine",
    commands: string[] = []
  ): Promise<DockerFile> => {
    setGenerating(true);
    toast.info("Generating Dockerfile...");
    
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    const dockerfile: DockerFile = {
      name: "Dockerfile",
      type: "dockerfile",
      content: `# Multi-stage build for Smart Friend Ultra
FROM ${baseImage} AS builder

WORKDIR /app

# Copy package files
COPY package*.json ./

# Install dependencies
RUN npm ci

# Copy source code
COPY . .

# Build application
${commands.map(cmd => `RUN ${cmd}`).join('\n')}

# Production stage
FROM ${baseImage}

WORKDIR /app

# Copy built application
COPY --from=builder /app/dist ./dist
COPY --from=builder /app/node_modules ./node_modules
COPY package*.json ./

EXPOSE 3000

CMD ["npm", "start"]
`
    };
    
    setDockerFiles(prev => [...prev, dockerfile]);
    setGenerating(false);
    toast.success("Dockerfile generated");
    return dockerfile;
  };

  const generateDockerCompose = async (): Promise<DockerFile> => {
    setGenerating(true);
    toast.info("Generating docker-compose.yml...");
    
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    const compose: DockerFile = {
      name: "docker-compose.yml",
      type: "compose",
      content: `version: '3.8'

services:
  app:
    build:
      context: .
      dockerfile: Dockerfile
    ports:
      - "3000:3000"
    environment:
      - NODE_ENV=production
      - VITE_SUPABASE_URL=\${VITE_SUPABASE_URL}
      - VITE_SUPABASE_PUBLISHABLE_KEY=\${VITE_SUPABASE_PUBLISHABLE_KEY}
    volumes:
      - ./data:/app/data
    restart: unless-stopped
    networks:
      - smart-friend-network

  nginx:
    image: nginx:alpine
    ports:
      - "80:80"
      - "443:443"
    volumes:
      - ./nginx.conf:/etc/nginx/nginx.conf
      - ./ssl:/etc/nginx/ssl
    depends_on:
      - app
    networks:
      - smart-friend-network

networks:
  smart-friend-network:
    driver: bridge
`
    };
    
    setDockerFiles(prev => [...prev, compose]);
    setGenerating(false);
    toast.success("docker-compose.yml generated");
    return compose;
  };

  const generateDockerIgnore = async (): Promise<DockerFile> => {
    setGenerating(true);
    
    const dockerignore: DockerFile = {
      name: ".dockerignore",
      type: "ignore",
      content: `# Dependencies
node_modules
npm-debug.log
yarn-debug.log
yarn-error.log

# Testing
coverage
.nyc_output

# Production
dist
build

# Misc
.DS_Store
.env.local
.env.development.local
.env.test.local
.env.production.local

# IDE
.idea
.vscode
*.swp
*.swo

# Git
.git
.gitignore
`
    };
    
    setDockerFiles(prev => [...prev, dockerignore]);
    setGenerating(false);
    return dockerignore;
  };

  return {
    generateDockerfile,
    generateDockerCompose,
    generateDockerIgnore,
    dockerFiles,
    generating
  };
};
