import { useCallback } from "react";

export type EmotionType = "neutral" | "happy" | "sad" | "angry" | "stressed" | "confused" | "excited" | "grateful";

export interface EmotionAnalysis {
  emotion: EmotionType;
  confidence: number;
  triggers: string[];
  suggestedTone: string;
}

export const useEmotionAI = () => {
  const analyzeEmotion = useCallback((text: string): EmotionAnalysis => {
    const lowerText = text.toLowerCase();
    
    // Happiness indicators
    const happyPatterns = [
      /thank/i, /thanks/i, /great/i, /awesome/i, /excellent/i, /love/i,
      /amazing/i, /wonderful/i, /fantastic/i, /ধন্যবাদ/i, /অসাধারণ/i, /চমৎকার/i
    ];
    
    // Sadness indicators
    const sadPatterns = [
      /sad/i, /depressed/i, /down/i, /unhappy/i, /disappointed/i,
      /দুঃখ/i, /বিষণ্ণ/i, /মন খারাপ/i
    ];
    
    // Anger indicators
    const angryPatterns = [
      /angry/i, /mad/i, /furious/i, /hate/i, /annoyed/i, /frustrated/i,
      /রাগ/i, /ক্রুদ্ধ/i, /বিরক্ত/i
    ];
    
    // Stress indicators
    const stressedPatterns = [
      /stress/i, /worried/i, /anxious/i, /nervous/i, /overwhelmed/i,
      /চাপ/i, /দুশ্চিন্তা/i, /উদ্বিগ্ন/i
    ];
    
    // Confusion indicators
    const confusedPatterns = [
      /confused/i, /don't understand/i, /unclear/i, /what do you mean/i,
      /বুঝিনি/i, /বুঝতে পারছি না/i, /কি বুঝাতে চাচ্ছ/i
    ];

    // Excitement indicators
    const excitedPatterns = [
      /excited/i, /can't wait/i, /wow/i, /amazing/i, /incredible/i,
      /উত্তেজিত/i, /দারুণ/i
    ];

    // Grateful indicators
    const gratefulPatterns = [
      /grateful/i, /appreciate/i, /thank you/i, /thanks so much/i,
      /কৃতজ্ঞ/i, /অনেক ধন্যবাদ/i
    ];

    // Check patterns
    let emotion: EmotionType = "neutral";
    let confidence = 50;
    const triggers: string[] = [];

    if (happyPatterns.some(p => p.test(lowerText))) {
      emotion = "happy";
      confidence = 75;
      triggers.push("positive words detected");
    } else if (sadPatterns.some(p => p.test(lowerText))) {
      emotion = "sad";
      confidence = 70;
      triggers.push("sadness indicators");
    } else if (angryPatterns.some(p => p.test(lowerText))) {
      emotion = "angry";
      confidence = 80;
      triggers.push("anger indicators");
    } else if (stressedPatterns.some(p => p.test(lowerText))) {
      emotion = "stressed";
      confidence = 72;
      triggers.push("stress indicators");
    } else if (confusedPatterns.some(p => p.test(lowerText))) {
      emotion = "confused";
      confidence = 75;
      triggers.push("confusion indicators");
    } else if (excitedPatterns.some(p => p.test(lowerText))) {
      emotion = "excited";
      confidence = 78;
      triggers.push("excitement indicators");
    } else if (gratefulPatterns.some(p => p.test(lowerText))) {
      emotion = "grateful";
      confidence = 82;
      triggers.push("gratitude indicators");
    }

    // Determine suggested tone
    const toneMap: Record<EmotionType, string> = {
      neutral: "professional and clear",
      happy: "warm and enthusiastic",
      sad: "empathetic and supportive",
      angry: "calm and understanding",
      stressed: "reassuring and patient",
      confused: "clear and explanatory",
      excited: "engaging and encouraging",
      grateful: "appreciative and friendly",
    };

    return {
      emotion,
      confidence,
      triggers,
      suggestedTone: toneMap[emotion],
    };
  }, []);

  const getEmotionColor = useCallback((emotion: EmotionType): string => {
    const colorMap: Record<EmotionType, string> = {
      neutral: "bg-muted text-muted-foreground",
      happy: "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200",
      sad: "bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200",
      angry: "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200",
      stressed: "bg-orange-100 text-orange-800 dark:bg-orange-900 dark:text-orange-200",
      confused: "bg-purple-100 text-purple-800 dark:bg-purple-900 dark:text-purple-200",
      excited: "bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200",
      grateful: "bg-pink-100 text-pink-800 dark:bg-pink-900 dark:text-pink-200",
    };
    return colorMap[emotion];
  }, []);

  const getEmotionLabel = useCallback((emotion: EmotionType, language: string = "en"): string => {
    const labelMap: Record<EmotionType, Record<string, string>> = {
      neutral: { en: "Neutral", bn: "নিরপেক্ষ" },
      happy: { en: "Happy", bn: "খুশি" },
      sad: { en: "Sad", bn: "দুঃখিত" },
      angry: { en: "Angry", bn: "রাগান্বিত" },
      stressed: { en: "Stressed", bn: "চাপগ্রস্ত" },
      confused: { en: "Confused", bn: "বিভ্রান্ত" },
      excited: { en: "Excited", bn: "উত্তেজিত" },
      grateful: { en: "Grateful", bn: "কৃতজ্ঞ" },
    };
    return labelMap[emotion][language] || labelMap[emotion].en;
  }, []);

  return {
    analyzeEmotion,
    getEmotionColor,
    getEmotionLabel,
  };
};
