import { useState } from "react";
import { toast } from "sonner";

export interface FTPCredentials {
  host: string;
  port: number;
  username: string;
  password: string;
  path: string;
  passive?: boolean;
}

export const useFTPDeploy = () => {
  const [connecting, setConnecting] = useState(false);
  const [connected, setConnected] = useState(false);

  const testConnection = async (credentials: FTPCredentials): Promise<boolean> => {
    setConnecting(true);
    toast.info("Testing FTP connection...");

    try {
      await new Promise(resolve => setTimeout(resolve, 1500));
      
      // Simulate connection test
      const success = Math.random() > 0.2; // 80% success rate
      
      if (success) {
        setConnected(true);
        toast.success("FTP connection successful!");
        return true;
      } else {
        throw new Error("Connection refused");
      }
    } catch (error: any) {
      toast.error(`FTP connection failed: ${error.message}`);
      return false;
    } finally {
      setConnecting(false);
    }
  };

  const uploadFile = async (
    credentials: FTPCredentials,
    localPath: string,
    remotePath: string
  ): Promise<boolean> => {
    toast.info("Uploading file via FTP...");

    try {
      await new Promise(resolve => setTimeout(resolve, 2000));
      toast.success("File uploaded successfully");
      return true;
    } catch (error: any) {
      toast.error(`Upload failed: ${error.message}`);
      return false;
    }
  };

  const uploadDirectory = async (
    credentials: FTPCredentials,
    localPath: string,
    remotePath: string
  ): Promise<boolean> => {
    toast.info("Uploading directory via FTP...");

    try {
      // Simulate progressive upload
      for (let i = 0; i <= 100; i += 20) {
        await new Promise(resolve => setTimeout(resolve, 500));
        if (i < 100) {
          toast.info(`Upload progress: ${i}%`);
        }
      }
      
      toast.success("Directory uploaded successfully");
      return true;
    } catch (error: any) {
      toast.error(`Upload failed: ${error.message}`);
      return false;
    }
  };

  const disconnect = () => {
    setConnected(false);
    toast.info("FTP connection closed");
  };

  return {
    testConnection,
    uploadFile,
    uploadDirectory,
    disconnect,
    connecting,
    connected
  };
};
