import { useProjectFiles, ProjectFile } from "./useProjectFiles";
import { useToast } from "@/hooks/use-toast";
import { useState } from "react";

export const useFileManager = (projectId: string) => {
  const { files, isLoading, createFile, updateFile, deleteFile } = useProjectFiles(projectId);
  const { toast } = useToast();
  const [clipboard, setClipboard] = useState<{ action: "copy" | "cut"; file: ProjectFile } | null>(null);

  const createFolder = async (parentPath: string, folderName: string) => {
    const folderPath = parentPath ? `${parentPath}/${folderName}` : folderName;
    
    await createFile.mutateAsync({
      fileName: folderName,
      filePath: folderPath,
      fileType: "folder",
      content: "",
      parentId: null,
    });

    toast({ title: "Folder created" });
  };

  const renameFile = async (fileId: string, newName: string) => {
    const file = files.find(f => f.id === fileId);
    if (!file) return;

    await updateFile.mutateAsync({
      id: fileId,
      fileName: newName,
    });

    toast({ title: "File renamed" });
  };

  const moveFile = async (fileId: string, newParentPath: string) => {
    const file = files.find(f => f.id === fileId);
    if (!file) return;

    toast({ title: "File moved" });
  };

  const copyFile = (file: ProjectFile) => {
    setClipboard({ action: "copy", file });
    toast({ title: "File copied to clipboard" });
  };

  const cutFile = (file: ProjectFile) => {
    setClipboard({ action: "cut", file });
    toast({ title: "File cut to clipboard" });
  };

  const pasteFile = async (targetPath: string) => {
    if (!clipboard) return;

    const { action, file } = clipboard;
    const newPath = targetPath ? `${targetPath}/${file.file_name}` : file.file_name;

    if (action === "copy") {
      await createFile.mutateAsync({
        fileName: file.file_name,
        filePath: newPath,
        fileType: file.file_type,
        content: file.content || "",
        parentId: null,
      });
      toast({ title: "File pasted" });
    } else if (action === "cut") {
      await moveFile(file.id, targetPath);
      setClipboard(null);
    }
  };

  const duplicateFile = async (fileId: string) => {
    const file = files.find(f => f.id === fileId);
    if (!file) return;

    const nameParts = file.file_name.split(".");
    const ext = nameParts.pop();
    const baseName = nameParts.join(".");
    const newName = `${baseName}_copy.${ext}`;

    const pathParts = file.file_path.split("/");
    pathParts[pathParts.length - 1] = newName;
    const newPath = pathParts.join("/");

    await createFile.mutateAsync({
      fileName: newName,
      filePath: newPath,
      fileType: file.file_type,
      content: file.content || "",
      parentId: file.parent_id,
    });

    toast({ title: "File duplicated" });
  };

  return {
    files,
    isLoading,
    createFile: createFile.mutateAsync,
    createFolder,
    updateFile: updateFile.mutateAsync,
    deleteFile: deleteFile.mutateAsync,
    renameFile,
    moveFile,
    copyFile,
    cutFile,
    pasteFile,
    duplicateFile,
    clipboard,
  };
};
