import { useState } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useToast } from "@/hooks/use-toast";

export interface BrainResponse {
  replyText: string;
  suggestedModule?: string | null;
  emotion?: string;
  confidence?: number;
}

export const useGlobalBrain = () => {
  const [isProcessing, setIsProcessing] = useState(false);
  const { toast } = useToast();

  const sendMessage = async (
    text: string,
    locale: string = "en",
    context?: any
  ): Promise<BrainResponse | null> => {
    setIsProcessing(true);
    try {
      const { data: { session } } = await supabase.auth.getSession();
      
      if (!session) {
        throw new Error("Not authenticated");
      }

      const { data, error } = await supabase.functions.invoke("brain-message", {
        body: { text, locale, context },
      });

      if (error) throw error;

      return data as BrainResponse;
    } catch (error: any) {
      console.error("Brain message error:", error);
      toast({
        title: "Error",
        description: error.message || "Failed to process message",
        variant: "destructive",
      });
      return null;
    } finally {
      setIsProcessing(false);
    }
  };

  const analyzeIntent = (text: string): {
    isQuestion: boolean;
    isCommand: boolean;
    topic: string | null;
  } => {
    const lowerText = text.toLowerCase();
    
    const isQuestion = 
      lowerText.includes("?") ||
      lowerText.startsWith("what") ||
      lowerText.startsWith("how") ||
      lowerText.startsWith("why") ||
      lowerText.startsWith("when") ||
      lowerText.startsWith("where") ||
      lowerText.startsWith("কি") ||
      lowerText.startsWith("কেন") ||
      lowerText.startsWith("কিভাবে");

    const isCommand = 
      lowerText.startsWith("create") ||
      lowerText.startsWith("make") ||
      lowerText.startsWith("generate") ||
      lowerText.startsWith("build") ||
      lowerText.startsWith("তৈরি") ||
      lowerText.startsWith("বানাও");

    let topic = null;
    if (lowerText.includes("education") || lowerText.includes("শিক্ষা")) topic = "education";
    else if (lowerText.includes("health") || lowerText.includes("স্বাস্থ্য")) topic = "health";
    else if (lowerText.includes("design") || lowerText.includes("ডিজাইন")) topic = "design";
    else if (lowerText.includes("marketing") || lowerText.includes("মার্কেটিং")) topic = "marketing";

    return { isQuestion, isCommand, topic };
  };

  return {
    sendMessage,
    analyzeIntent,
    isProcessing,
  };
};
