import { useState, useCallback, useEffect } from "react";
import { supabase } from "@/integrations/supabase/client";
import { useOfflineBrain } from "./useOfflineBrain";
import { useOnlineStatus } from "./useOnlineStatus";
import { useToast } from "./use-toast";

export type BrainMode = "online" | "offline" | "hybrid";
export type BrainStatus = "idle" | "processing" | "failed" | "switching";

interface HybridBrainConfig {
  fallbackTimeout: number; // ms before falling back to offline
  retryAttempts: number;
  cacheEnabled: boolean;
  autoSwitchMode: boolean;
}

interface BrainResponse {
  answer: string;
  source: "online" | "offline" | "cache";
  confidence: number;
  processingTime: number;
  mode: BrainMode;
}

interface MemorySnapshot {
  id: string;
  context: string;
  timestamp: Date;
  embedding?: number[];
}

const DEFAULT_CONFIG: HybridBrainConfig = {
  fallbackTimeout: 5000,
  retryAttempts: 2,
  cacheEnabled: true,
  autoSwitchMode: true,
};

export const useHybridBrain = (config: Partial<HybridBrainConfig> = {}) => {
  const finalConfig = { ...DEFAULT_CONFIG, ...config };
  const { toast } = useToast();
  const isOnline = useOnlineStatus();
  const { searchOfflineResponse, saveOfflineResponse } = useOfflineBrain();
  
  const [mode, setMode] = useState<BrainMode>("hybrid");
  const [status, setStatus] = useState<BrainStatus>("idle");
  const [lastError, setLastError] = useState<string | null>(null);
  const [memorySnapshots, setMemorySnapshots] = useState<MemorySnapshot[]>([]);
  const [failureCount, setFailureCount] = useState(0);

  // Auto-switch mode based on online status
  useEffect(() => {
    if (finalConfig.autoSwitchMode) {
      if (!isOnline && mode === "online") {
        setMode("offline");
        toast({
          title: "Switched to Offline Mode",
          description: "Using local AI brain for responses",
        });
      } else if (isOnline && mode === "offline") {
        setMode("hybrid");
        toast({
          title: "Back Online",
          description: "Hybrid mode activated",
        });
      }
    }
  }, [isOnline, finalConfig.autoSwitchMode]);

  // Check knowledge cache
  const checkKnowledgeCache = useCallback(async (question: string): Promise<string | null> => {
    if (!finalConfig.cacheEnabled) return null;

    try {
      const cached = localStorage.getItem(`knowledge_cache_${question.toLowerCase()}`);
      if (cached) {
        const parsed = JSON.parse(cached);
        const age = Date.now() - parsed.timestamp;
        // Cache valid for 24 hours
        if (age < 24 * 60 * 60 * 1000) {
          return parsed.answer;
        }
      }
    } catch (error) {
      console.error("Cache check error:", error);
    }
    return null;
  }, [finalConfig.cacheEnabled]);

  // Save to knowledge cache
  const saveToCache = useCallback((question: string, answer: string) => {
    if (!finalConfig.cacheEnabled) return;

    try {
      localStorage.setItem(
        `knowledge_cache_${question.toLowerCase()}`,
        JSON.stringify({
          answer,
          timestamp: Date.now(),
        })
      );
    } catch (error) {
      console.error("Cache save error:", error);
    }
  }, [finalConfig.cacheEnabled]);

  // Online AI query with timeout
  const queryOnlineAI = useCallback(async (
    messages: any[],
    language: string,
    timeoutMs: number
  ): Promise<string | null> => {
    const controller = new AbortController();
    const timeout = setTimeout(() => controller.abort(), timeoutMs);

    try {
      const { data, error } = await supabase.functions.invoke("smart-friend-chat", {
        body: { messages, language },
      });

      clearTimeout(timeout);

      if (error) throw error;
      return data?.response || null;
    } catch (error: any) {
      clearTimeout(timeout);
      if (error.name === 'AbortError') {
        console.log("Online AI timeout");
        return null;
      }
      console.error("Online AI error:", error);
      return null;
    }
  }, []);

  // Main hybrid query function
  const query = useCallback(async (
    question: string,
    messages: any[] = [],
    language: string = "bn"
  ): Promise<BrainResponse> => {
    const startTime = Date.now();
    setStatus("processing");
    setLastError(null);

    try {
      // Step 1: Check cache first
      const cachedAnswer = await checkKnowledgeCache(question);
      if (cachedAnswer) {
        setStatus("idle");
        return {
          answer: cachedAnswer,
          source: "cache",
          confidence: 1.0,
          processingTime: Date.now() - startTime,
          mode: "hybrid",
        };
      }

      // Step 2: Try online if available
      if (mode !== "offline" && isOnline) {
        const onlineAnswer = await queryOnlineAI(messages, language, finalConfig.fallbackTimeout);
        
        if (onlineAnswer) {
          setStatus("idle");
          setFailureCount(0);
          saveToCache(question, onlineAnswer);
          saveOfflineResponse(question, onlineAnswer);
          
          return {
            answer: onlineAnswer,
            source: "online",
            confidence: 0.95,
            processingTime: Date.now() - startTime,
            mode: "online",
          };
        } else {
          setFailureCount(prev => prev + 1);
        }
      }

      // Step 3: Fallback to offline
      setStatus("switching");
      const offlineAnswer = await searchOfflineResponse(question);
      
      if (offlineAnswer) {
        setStatus("idle");
        return {
          answer: offlineAnswer,
          source: "offline",
          confidence: 0.85,
          processingTime: Date.now() - startTime,
          mode: "offline",
        };
      }

      // Step 4: Generate fallback response
      setStatus("idle");
      const fallbackAnswer = language === "bn" 
        ? "দুঃখিত, আমি এই মুহূর্তে আপনার প্রশ্নের উত্তর দিতে পারছি না। অনুগ্রহ করে পরে আবার চেষ্টা করুন।"
        : "I'm sorry, I cannot answer your question at the moment. Please try again later.";

      return {
        answer: fallbackAnswer,
        source: "offline",
        confidence: 0.5,
        processingTime: Date.now() - startTime,
        mode: "offline",
      };

    } catch (error: any) {
      console.error("Hybrid brain error:", error);
      setStatus("failed");
      setLastError(error.message);
      
      const errorAnswer = language === "bn"
        ? "একটি ত্রুটি ঘটেছে। অনুগ্রহ করে আবার চেষ্টা করুন।"
        : "An error occurred. Please try again.";

      return {
        answer: errorAnswer,
        source: "offline",
        confidence: 0.3,
        processingTime: Date.now() - startTime,
        mode: "offline",
      };
    }
  }, [mode, isOnline, checkKnowledgeCache, queryOnlineAI, searchOfflineResponse, saveOfflineResponse, saveToCache, finalConfig.fallbackTimeout]);

  // Create memory snapshot
  const createMemorySnapshot = useCallback((context: string) => {
    const snapshot: MemorySnapshot = {
      id: Date.now().toString(),
      context,
      timestamp: new Date(),
    };

    setMemorySnapshots(prev => [...prev.slice(-9), snapshot]); // Keep last 10
    
    // Save to localStorage
    try {
      localStorage.setItem("memory_snapshots", JSON.stringify([...memorySnapshots.slice(-9), snapshot]));
    } catch (error) {
      console.error("Failed to save memory snapshot:", error);
    }
  }, [memorySnapshots]);

  // Load memory snapshots
  useEffect(() => {
    try {
      const stored = localStorage.getItem("memory_snapshots");
      if (stored) {
        setMemorySnapshots(JSON.parse(stored));
      }
    } catch (error) {
      console.error("Failed to load memory snapshots:", error);
    }
  }, []);

  // Regenerate context from snapshots
  const regenerateContext = useCallback((): string => {
    if (memorySnapshots.length === 0) return "";
    
    const recentSnapshots = memorySnapshots.slice(-3);
    return recentSnapshots.map(s => s.context).join("\n\n");
  }, [memorySnapshots]);

  // Clear memory
  const clearMemory = useCallback(() => {
    setMemorySnapshots([]);
    localStorage.removeItem("memory_snapshots");
  }, []);

  return {
    query,
    mode,
    setMode,
    status,
    lastError,
    failureCount,
    isOnline,
    memorySnapshots,
    createMemorySnapshot,
    regenerateContext,
    clearMemory,
  };
};