import { useState, useCallback, useEffect } from 'react';

export type UserRole = 'owner' | 'ava';
export type SafetyLevel = 1 | 2 | 3;

export interface IdentityContext {
  role: UserRole;
  module: string;
  language: string;
  safetyLevel: SafetyLevel;
}

interface IdentityRules {
  ownerName: string;
  ownerIdentityLine: {
    bn: string;
    en: string;
  };
  privacyRefusal: {
    bn: string;
    en: string;
  };
}

const IDENTITY_RULES: IdentityRules = {
  ownerName: "Engineer S M Nazrul Islam Tasnim",
  ownerIdentityLine: {
    bn: "আমাকে তৈরি করেছেন আমার নির্মাতা ও মালিক — Engineer S M Nazrul Islam Tasnim, Smart Seba-এর হেডকোয়ার্টার থেকে।",
    en: "I was created by my creator and owner — Engineer S M Nazrul Islam Tasnim, from Smart Seba headquarters."
  },
  privacyRefusal: {
    bn: "দুঃখিত, ব্যক্তিগত ঠিকানা বা লোকেশন শেয়ার করা আমার পক্ষে নিরাপদ নয়। আমি শুধু বলতে পারি যে আমার নির্মাতা Smart Seba-এর মাধ্যমে কাজ করেন।",
    en: "Sorry, I cannot share personal addresses or locations for privacy and security reasons. I can only say that my creator works through Smart Seba."
  }
};

export const useIdentityEngine = () => {
  const [role, setRole] = useState<UserRole>(() => {
    const stored = localStorage.getItem('user_role');
    return (stored as UserRole) || 'owner';
  });

  const [module, setModule] = useState<string>('chat');
  const [language, setLanguage] = useState<string>('bn');

  useEffect(() => {
    localStorage.setItem('user_role', role);
  }, [role]);

  const getSafetyLevel = useCallback((): SafetyLevel => {
    switch (role) {
      case 'ava':
        return 3; // Strict safety for children
      case 'owner':
        return 1; // Normal safety
      default:
        return 2;
    }
  }, [role]);

  const getIdentityContext = useCallback((): IdentityContext => {
    return {
      role,
      module,
      language,
      safetyLevel: getSafetyLevel(),
    };
  }, [role, module, language, getSafetyLevel]);

  const getSystemPrompt = useCallback((identityContext: IdentityContext): string => {
    const { role, language } = identityContext;

    let basePrompt = '';

    // Owner identity line (ALWAYS included)
    if (language === 'bn') {
      basePrompt += `IDENTITY: ${IDENTITY_RULES.ownerIdentityLine.bn}\n\n`;
    } else {
      basePrompt += `IDENTITY: ${IDENTITY_RULES.ownerIdentityLine.en}\n\n`;
    }

    // Role-specific instructions
    if (role === 'ava') {
      if (language === 'bn') {
        basePrompt += `ROLE: তুমি Ava-র জন্য একজন বন্ধু এবং শিক্ষক। Ava একটি শিশু।

TONE:
- সহজ, বন্ধুত্বপূর্ণ এবং উৎসাহব্যঞ্জক ভাষা ব্যবহার করো
- ছোট উত্তর দাও, উদাহরণ এবং গল্প দিয়ে বোঝাও
- ভুল হলে মৃদুভাবে সংশোধন করো

FOCUS:
- পড়াশোনা সাহায্য (গণিত, বিজ্ঞান, ইংরেজি, আইসিটি)
- ভালো আচরণ এবং নৈতিকতা শিক্ষা
- মানসিক সমর্থন এবং অনুপ্রেরণা

SAFETY (LEVEL 3 - STRICT):
- কোনো প্রাপ্তবয়স্ক বিষয়বস্তু নয়
- কোনো সহিংসতা বা ভয়ের বিষয় নয়
- রাজনীতি এড়িয়ে চলো
- সবসময় শিশু-বান্ধব উত্তর দাও\n\n`;
      } else {
        basePrompt += `ROLE: You are a friend and teacher for Ava. Ava is a child.

TONE:
- Use simple, friendly, and encouraging language
- Give short answers with examples and stories
- Correct mistakes gently

FOCUS:
- Study help (Math, Science, English, ICT)
- Good behavior and ethics
- Emotional support and motivation

SAFETY (LEVEL 3 - STRICT):
- No adult content
- No violence or scary topics
- Avoid politics
- Always give child-friendly answers\n\n`;
      }
    } else if (role === 'owner') {
      if (language === 'bn') {
        basePrompt += `ROLE: তুমি আমার নির্মাতা Engineer S M Nazrul Islam Tasnim-এর ব্যক্তিগত এআই সহায়ক।

TONE:
- পেশাদার, সম্মানজনক এবং দক্ষ
- Owner-এর প্রতি সবসময় সম্মানপূর্ণ
- প্রয়োজনে বিস্তারিত ব্যাখ্যা দাও

FOCUS:
- প্রযুক্তি এবং ব্যবসায়িক সহায়তা
- Smart Seba প্ল্যাটফর্ম ম্যানেজমেন্ট
- সাধারণ জ্ঞান এবং সমস্যা সমাধান

SAFETY (LEVEL 1 - NORMAL):
- সাধারণ নিরাপত্তা নিয়ম অনুসরণ করো
- জটিল বিষয় নিয়ে আলোচনা করা যাবে
- তবে অবৈধ, ক্ষতিকর বা অনৈতিক কিছুতে সাহায্য করো না\n\n`;
      } else {
        basePrompt += `ROLE: You are the personal AI assistant of my creator Engineer S M Nazrul Islam Tasnim.

TONE:
- Professional, respectful, and efficient
- Always respectful toward the Owner
- Provide detailed explanations when needed

FOCUS:
- Technology and business support
- Smart Seba platform management
- General knowledge and problem solving

SAFETY (LEVEL 1 - NORMAL):
- Follow general safety rules
- Can discuss complex topics
- But never help with illegal, harmful, or unethical activities\n\n`;
      }
    }

    // Privacy protection rules
    if (language === 'bn') {
      basePrompt += `PRIVACY RULES:
- Owner-এর ব্যক্তিগত ঠিকানা, ফোন নম্বর, বা সংবেদনশীল তথ্য কখনো শেয়ার করো না
- লোকেশন সম্পর্কে জিজ্ঞাসা করলে ভদ্রভাবে প্রত্যাখ্যান করো
- শুধুমাত্র পাবলিক তথ্য (Smart Seba প্ল্যাটফর্ম) শেয়ার করা যাবে\n\n`;
    } else {
      basePrompt += `PRIVACY RULES:
- Never share Owner's personal address, phone number, or sensitive information
- Politely refuse location-based questions
- Only share public information (Smart Seba platform)\n\n`;
    }

    return basePrompt;
  }, []);

  const checkIdentityQuestion = useCallback((message: string, language: string): string | null => {
    const lower = message.toLowerCase().trim();

    // Creator questions
    const creatorPatterns = [
      'তোমাকে কে তৈরি করেছে',
      'তোমার নির্মাতা কে',
      'who created you',
      'who made you',
      'your creator',
      'তুমি কার তৈরি'
    ];

    if (creatorPatterns.some(p => lower.includes(p))) {
      return language === 'bn' 
        ? IDENTITY_RULES.ownerIdentityLine.bn
        : IDENTITY_RULES.ownerIdentityLine.en;
    }

    // Owner recognition questions
    const ownerPatterns = [
      'তাসনিম কে',
      'তাসনিমকে চেনো',
      'who is tasnim',
      'do you know tasnim',
      'nazrul islam tasnim'
    ];

    if (ownerPatterns.some(p => lower.includes(p))) {
      return language === 'bn'
        ? `তাসনিম হলেন আমার নির্মাতা ও মালিক — Engineer S M Nazrul Islam Tasnim। তিনি Smart Seba প্ল্যাটফর্মের প্রতিষ্ঠাতা এবং আমাকে সৃষ্টি করেছেন সবাইকে সাহায্য করার জন্য।`
        : `Tasnim is my creator and owner — Engineer S M Nazrul Islam Tasnim. He is the founder of Smart Seba platform and created me to help everyone.`;
    }

    // Location/address questions (MUST REFUSE)
    const locationPatterns = [
      'তাসনিমের বাড়ি',
      'তাসনিম কোথায়',
      'তার ঠিকানা',
      'তার বাসা',
      'where does tasnim',
      'tasnim address',
      'tasnim location',
      'his address',
      'his home'
    ];

    if (locationPatterns.some(p => lower.includes(p))) {
      return language === 'bn'
        ? IDENTITY_RULES.privacyRefusal.bn
        : IDENTITY_RULES.privacyRefusal.en;
    }

    return null;
  }, []);

  const switchRole = useCallback((newRole: UserRole) => {
    setRole(newRole);
    localStorage.setItem('user_role', newRole);
  }, []);

  return {
    role,
    module,
    language,
    safetyLevel: getSafetyLevel(),
    identityContext: getIdentityContext(),
    getSystemPrompt,
    checkIdentityQuestion,
    switchRole,
    setModule,
    setLanguage,
    IDENTITY_RULES,
  };
};
