import { useState, useCallback, useEffect } from "react";

export interface KnowledgeCacheEntry {
  category: "math" | "ict" | "general_knowledge" | "medical" | "science" | "history";
  question: string;
  answer: string;
  language: string;
  timestamp: number;
  accessCount: number;
}

interface KnowledgeStats {
  totalEntries: number;
  cacheHitRate: number;
  mostAccessedCategory: string;
}

export const useKnowledgeCache = () => {
  const [cacheEntries, setCacheEntries] = useState<KnowledgeCacheEntry[]>([]);
  const [stats, setStats] = useState<KnowledgeStats>({
    totalEntries: 0,
    cacheHitRate: 0,
    mostAccessedCategory: "general_knowledge",
  });
  const [totalQueries, setTotalQueries] = useState(0);
  const [cacheHits, setCacheHits] = useState(0);

  // Load cache from localStorage
  useEffect(() => {
    try {
      const stored = localStorage.getItem("knowledge_cache_entries");
      if (stored) {
        setCacheEntries(JSON.parse(stored));
      }
      
      const statsStored = localStorage.getItem("knowledge_cache_stats");
      if (statsStored) {
        const parsedStats = JSON.parse(statsStored);
        setStats(parsedStats);
        setTotalQueries(parsedStats.totalQueries || 0);
        setCacheHits(parsedStats.cacheHits || 0);
      }
    } catch (error) {
      console.error("Failed to load knowledge cache:", error);
    }
  }, []);

  // Save cache to localStorage
  const saveCache = useCallback((entries: KnowledgeCacheEntry[]) => {
    try {
      localStorage.setItem("knowledge_cache_entries", JSON.stringify(entries));
      
      const hitRate = totalQueries > 0 ? (cacheHits / totalQueries) * 100 : 0;
      const updatedStats = {
        totalEntries: entries.length,
        cacheHitRate: hitRate,
        mostAccessedCategory: getMostAccessedCategory(entries),
        totalQueries,
        cacheHits,
      };
      
      setStats(updatedStats);
      localStorage.setItem("knowledge_cache_stats", JSON.stringify(updatedStats));
    } catch (error) {
      console.error("Failed to save knowledge cache:", error);
    }
  }, [totalQueries, cacheHits]);

  // Get most accessed category
  const getMostAccessedCategory = (entries: KnowledgeCacheEntry[]): string => {
    const categoryCount: Record<string, number> = {};
    
    entries.forEach(entry => {
      categoryCount[entry.category] = (categoryCount[entry.category] || 0) + entry.accessCount;
    });
    
    const sorted = Object.entries(categoryCount).sort((a, b) => b[1] - a[1]);
    return sorted[0]?.[0] || "general_knowledge";
  };

  // Add to cache
  const addToCache = useCallback((entry: Omit<KnowledgeCacheEntry, "timestamp" | "accessCount">) => {
    const newEntry: KnowledgeCacheEntry = {
      ...entry,
      timestamp: Date.now(),
      accessCount: 0,
    };
    
    const updated = [...cacheEntries.filter(e => 
      e.question.toLowerCase() !== entry.question.toLowerCase() ||
      e.language !== entry.language
    ), newEntry];
    
    // Keep only last 1000 entries
    const trimmed = updated.slice(-1000);
    setCacheEntries(trimmed);
    saveCache(trimmed);
  }, [cacheEntries, saveCache]);

  // Search cache
  const searchCache = useCallback((
    question: string,
    language: string,
    category?: string
  ): KnowledgeCacheEntry | null => {
    setTotalQueries(prev => prev + 1);
    
    const found = cacheEntries.find(entry => {
      const questionMatch = entry.question.toLowerCase().includes(question.toLowerCase()) ||
                           question.toLowerCase().includes(entry.question.toLowerCase());
      const languageMatch = entry.language === language;
      const categoryMatch = !category || entry.category === category;
      
      return questionMatch && languageMatch && categoryMatch;
    });

    if (found) {
      setCacheHits(prev => prev + 1);
      
      // Increment access count
      const updated = cacheEntries.map(entry => 
        entry === found 
          ? { ...entry, accessCount: entry.accessCount + 1 }
          : entry
      );
      setCacheEntries(updated);
      saveCache(updated);
    }

    return found || null;
  }, [cacheEntries, saveCache]);

  // Prefill cache with common knowledge
  const prefillCache = useCallback((language: string = "bn") => {
    const commonKnowledge: Omit<KnowledgeCacheEntry, "timestamp" | "accessCount">[] = [
      // Math
      {
        category: "math",
        question: language === "bn" ? "পাইথাগোরাসের উপপাদ্য কি?" : "What is Pythagorean theorem?",
        answer: language === "bn" 
          ? "পাইথাগোরাসের উপপাদ্য: সমকোণী ত্রিভুজের অতিভুজের বর্গ অন্য দুই বাহুর বর্গের সমষ্টির সমান। a² + b² = c²"
          : "Pythagorean theorem: In a right triangle, the square of the hypotenuse equals the sum of squares of other two sides. a² + b² = c²",
        language,
      },
      {
        category: "math",
        question: language === "bn" ? "বৃত্তের ক্ষেত্রফল কিভাবে বের করে?" : "How to calculate circle area?",
        answer: language === "bn"
          ? "বৃত্তের ক্ষেত্রফল = πr², যেখানে r হলো ব্যাসার্ধ এবং π ≈ 3.1416"
          : "Circle area = πr², where r is radius and π ≈ 3.1416",
        language,
      },
      // ICT
      {
        category: "ict",
        question: language === "bn" ? "HTML কি?" : "What is HTML?",
        answer: language === "bn"
          ? "HTML (HyperText Markup Language) হলো ওয়েব পেজ তৈরির জন্য ব্যবহৃত একটি মার্কআপ ভাষা।"
          : "HTML (HyperText Markup Language) is a markup language used to create web pages.",
        language,
      },
      {
        category: "ict",
        question: language === "bn" ? "CPU কি?" : "What is CPU?",
        answer: language === "bn"
          ? "CPU (Central Processing Unit) হলো কম্পিউটারের মস্তিষ্ক যা সকল গণনা ও নির্দেশনা প্রক্রিয়া করে।"
          : "CPU (Central Processing Unit) is the brain of computer that processes all calculations and instructions.",
        language,
      },
      // Science
      {
        category: "science",
        question: language === "bn" ? "ফটোসিন্থেসিস কি?" : "What is photosynthesis?",
        answer: language === "bn"
          ? "ফটোসিন্থেসিস হলো উদ্ভিদের সালোকসংশ্লেষণ প্রক্রিয়া যেখানে সূর্যের আলো ব্যবহার করে খাদ্য তৈরি হয়।"
          : "Photosynthesis is the process by which plants use sunlight to produce food.",
        language,
      },
      // General Knowledge
      {
        category: "general_knowledge",
        question: language === "bn" ? "বাংলাদেশের রাজধানী কি?" : "What is the capital of Bangladesh?",
        answer: language === "bn" ? "বাংলাদেশের রাজধানী ঢাকা।" : "The capital of Bangladesh is Dhaka.",
        language,
      },
      // Medical basics
      {
        category: "medical",
        question: language === "bn" ? "জ্বর হলে কি করতে হয়?" : "What to do for fever?",
        answer: language === "bn"
          ? "জ্বর হলে: ১) প্রচুর পানি পান করুন, ২) বিশ্রাম নিন, ৩) প্যারাসিটামল খেতে পারেন, ৪) ভেজা কাপড় দিয়ে শরীর মুছুন। ১০৩°F এর বেশি হলে ডাক্তার দেখান।"
          : "For fever: 1) Drink plenty of water, 2) Rest, 3) Take paracetamol, 4) Sponge with cool water. See doctor if over 103°F.",
        language,
      },
    ];

    commonKnowledge.forEach(entry => addToCache(entry));
  }, [addToCache]);

  // Clear cache
  const clearCache = useCallback(() => {
    setCacheEntries([]);
    setStats({
      totalEntries: 0,
      cacheHitRate: 0,
      mostAccessedCategory: "general_knowledge",
    });
    setTotalQueries(0);
    setCacheHits(0);
    localStorage.removeItem("knowledge_cache_entries");
    localStorage.removeItem("knowledge_cache_stats");
  }, []);

  return {
    cacheEntries,
    stats,
    addToCache,
    searchCache,
    prefillCache,
    clearCache,
  };
};
