import { createContext, useContext, useState, useEffect, ReactNode } from "react";
import { supabase } from "@/integrations/supabase/client";
import { detectBrowserLanguage, isRTL, getLanguage } from "@/lib/i18n/languages";
import { translate } from "@/lib/i18n/translations";

export type Language = string; // Support 100+ languages

interface LanguageContextType {
  language: Language;
  setLanguage: (lang: Language) => void;
  t: (key: string, fallback?: string) => string;
  direction: 'ltr' | 'rtl';
  voiceCode: string;
}

const LanguageContext = createContext<LanguageContextType | undefined>(undefined);

export const LanguageProvider = ({ children }: { children: ReactNode }) => {
  const [language, setLanguageState] = useState<Language>("en");
  const [direction, setDirection] = useState<'ltr' | 'rtl'>('ltr');

  useEffect(() => {
    // Load language preference
    const loadLanguage = async () => {
      const { data: { user } } = await supabase.auth.getUser();
      if (user) {
        const { data } = await supabase
          .from("profiles")
          .select("language_preference")
          .eq("id", user.id)
          .single();
        
        if (data?.language_preference) {
          const lang = data.language_preference as Language;
          setLanguageState(lang);
          setDirection(isRTL(lang) ? 'rtl' : 'ltr');
          // Apply direction to document
          document.documentElement.dir = isRTL(lang) ? 'rtl' : 'ltr';
          document.documentElement.lang = lang;
        }
      } else {
        // Auto-detect browser language
        const detectedLang = detectBrowserLanguage();
        setLanguageState(detectedLang);
        setDirection(isRTL(detectedLang) ? 'rtl' : 'ltr');
        document.documentElement.dir = isRTL(detectedLang) ? 'rtl' : 'ltr';
        document.documentElement.lang = detectedLang;
      }
    };

    loadLanguage();
  }, []);

  const setLanguage = async (lang: Language) => {
    setLanguageState(lang);
    const newDirection = isRTL(lang) ? 'rtl' : 'ltr';
    setDirection(newDirection);
    
    // Apply direction and language to document
    document.documentElement.dir = newDirection;
    document.documentElement.lang = lang;
    
    // Save to database if logged in
    const { data: { user } } = await supabase.auth.getUser();
    if (user) {
      await supabase
        .from("profiles")
        .update({ language_preference: lang })
        .eq("id", user.id);
    }
  };

  const t = (key: string, fallback?: string) => {
    return translate(key, language, fallback);
  };

  const langData = getLanguage(language);
  const voiceCode = langData?.voiceCode || 'en-US';

  return (
    <LanguageContext.Provider value={{ language, setLanguage, t, direction, voiceCode }}>
      {children}
    </LanguageContext.Provider>
  );
};

export const useLanguage = () => {
  const context = useContext(LanguageContext);
  if (context === undefined) {
    throw new Error("useLanguage must be used within a LanguageProvider");
  }
  return context;
};
