import { useState, useCallback } from "react";
import { useToast } from "@/hooks/use-toast";

export interface LaravelComponent {
  type: "controller" | "model" | "migration" | "route" | "policy" | "service";
  name: string;
  code: string;
  path: string;
}

export interface LaravelProject {
  name: string;
  components: LaravelComponent[];
  created: string;
}

export const useLaravelGenerator = () => {
  const [generating, setGenerating] = useState(false);
  const [projects, setProjects] = useState<LaravelProject[]>([]);
  const { toast } = useToast();

  const generateController = useCallback((name: string, methods: string[]) => {
    const code = `<?php

namespace App\\Http\\Controllers;

use Illuminate\\Http\\Request;
use Illuminate\\Http\\JsonResponse;

class ${name}Controller extends Controller
{
${methods.map(method => `    public function ${method}(Request $request): JsonResponse
    {
        // TODO: Implement ${method} logic
        return response()->json(['message' => '${method} executed']);
    }
`).join('\n')}
}`;

    return {
      type: "controller" as const,
      name: `${name}Controller`,
      code,
      path: `backend/app/Http/Controllers/${name}Controller.php`,
    };
  }, []);

  const generateModel = useCallback((name: string, fields: string[]) => {
    const code = `<?php

namespace App\\Models;

use Illuminate\\Database\\Eloquent\\Model;
use Illuminate\\Database\\Eloquent\\Factories\\HasFactory;

class ${name} extends Model
{
    use HasFactory;

    protected $fillable = [
${fields.map(f => `        '${f}'`).join(',\n')}
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];
}`;

    return {
      type: "model" as const,
      name,
      code,
      path: `backend/app/Models/${name}.php`,
    };
  }, []);

  const generateMigration = useCallback((table: string, fields: { name: string; type: string }[]) => {
    const timestamp = new Date().toISOString().replace(/[-:]/g, '').split('.')[0];
    const code = `<?php

use Illuminate\\Database\\Migrations\\Migration;
use Illuminate\\Database\\Schema\\Blueprint;
use Illuminate\\Support\\Facades\\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('${table}', function (Blueprint $table) {
            $table->id();
${fields.map(f => `            $table->${f.type}('${f.name}');`).join('\n')}
            $table->timestamps();
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('${table}');
    }
};`;

    return {
      type: "migration" as const,
      name: `create_${table}_table`,
      code,
      path: `backend/database/migrations/${timestamp}_create_${table}_table.php`,
    };
  }, []);

  const generateAPI = useCallback(async (config: {
    name: string;
    resource: string;
    fields: { name: string; type: string }[];
  }) => {
    setGenerating(true);
    try {
      await new Promise(resolve => setTimeout(resolve, 1500));

      const components: LaravelComponent[] = [];

      // Generate Model
      components.push(generateModel(
        config.resource,
        config.fields.map(f => f.name)
      ));

      // Generate Migration
      components.push(generateMigration(
        config.resource.toLowerCase() + 's',
        config.fields
      ));

      // Generate Controller
      components.push(generateController(
        config.resource,
        ['index', 'store', 'show', 'update', 'destroy']
      ));

      // Generate Route
      const routeCode = `<?php

use Illuminate\\Support\\Facades\\Route;
use App\\Http\\Controllers\\${config.resource}Controller;

Route::prefix('api')->group(function () {
    Route::apiResource('${config.resource.toLowerCase()}s', ${config.resource}Controller::class);
});`;

      components.push({
        type: "route",
        name: "api",
        code: routeCode,
        path: "backend/routes/api.php",
      });

      const project: LaravelProject = {
        name: config.name,
        components,
        created: new Date().toISOString(),
      };

      setProjects(prev => [...prev, project]);

      toast({
        title: "Laravel API Generated",
        description: `Created ${components.length} files`,
      });

      return project;
    } catch (error: any) {
      toast({
        title: "Generation Failed",
        description: error.message,
        variant: "destructive",
      });
      return null;
    } finally {
      setGenerating(false);
    }
  }, [generateController, generateModel, generateMigration, toast]);

  const exportBackend = useCallback(async (projectName: string) => {
    try {
      // Use existing ZIP export functionality
      const JSZip = (await import("jszip")).default;
      const zip = new JSZip();

      const project = projects.find(p => p.name === projectName);
      if (!project) throw new Error("Project not found");

      // Add all Laravel files
      project.components.forEach(component => {
        zip.file(component.path, component.code);
      });

      // Add composer.json
      const composerJson = {
        name: `smartfriend/${projectName.toLowerCase()}`,
        description: "Laravel API generated by Smart Friend",
        require: {
          "php": "^8.2",
          "laravel/framework": "^11.0"
        }
      };
      zip.file("backend/composer.json", JSON.stringify(composerJson, null, 2));

      // Generate and download
      const blob = await zip.generateAsync({ type: "blob" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = `${projectName}-backend.zip`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);

      toast({
        title: "Backend Exported",
        description: "Laravel project downloaded",
      });

      return true;
    } catch (error: any) {
      toast({
        title: "Export Failed",
        description: error.message,
        variant: "destructive",
      });
      return false;
    }
  }, [projects, toast]);

  return {
    generateAPI,
    exportBackend,
    generateController,
    generateModel,
    generateMigration,
    projects,
    generating,
  };
};
